package IMPL::ORM;
use strict;
use warnings;

use base qw(IMPL::Object);
use IMPL::Class::Property;
use Scalar::Util qw(weaken refaddr);

use IMPL::Exception;

our $Depth = 1; #   + 1  
our $UseProxy = 1;

BEGIN {
    private property _ObjectCache => prop_all;
    private property _MapInstances => prop_all;
    private property _WorkUnit => prop_all;
    public property Schema => prop_all;
}

sub ObjectInfoById {
    my ($this,$oid) = @_;
    
    return $this->_ObjectCache->{$oid};
}

sub ObjectInfo {
    my ($this,$inst) = @_;
    
    die new IMPL::InvalidOperationException("This method can be used only for a reference") unless ref $inst;
    
    return $this->_MapInstances->{refaddr $inst};
}


1;
__END__
=pod
=head1 SYNOPSIS

use IMPL::ORM::Sql;

my $DB = new IMPL::ORM::Sql("connection string");

local $IMPL::ORM::Depth = 1; # load childs only, no more

my $artist = $DB->Lookup( Artist => { name => 'Beatles' } );

my $order = new Order();
$order->AddItem($_) foreach $artist->Albums->List;

$DB->Save($order);

my $label = $artist->Albums->Index(0)->Label;

$DB->Populate($label); #load $label

=head1 DESCRIPTION
=head2 MEMBERS
=level 4
=back
=head2 Variables
=head2 INTERNALS
=head3 Object Representation

         
,       .

Foo         entityFoo
    Bar         entityBar
    Baz         entityBaz

        
  -      ,  - 
  .

Foo
    public virtual property Name => prop_all, {Type => String};
    
entityFoo
    string m_Name -  
    string v_Name -  
    
       .
    ,        .

=head3 Object Cache

  ,       
.      ,    
.

ObjectInfo => {
    instance => weak ref
    _id => data source dependent id
    state => {persistent|null|new|deleted}
    work_unit => ref to the work unit where object is acting
}

      ObjectInfoById  ObjectInfo

=head3 Type mapping

      ,    ,
   .     :

=level 4

=item 1

 ,       

=item

     ( ,   ..)

=item

,       ,   .

=back

=cut