package Test::DOM::Node;
use strict;
use warnings;

use parent qw(IMPL::Test::Unit);
use IMPL::Test qw(test shared failed cmparray assert);
use IMPL::Class::Property;
use Scalar::Util qw(weaken);

require IMPL::DOM::Node;

__PACKAGE__->PassThroughArgs;

BEGIN {
    shared public property Root => prop_all;
}

test Create => sub {
    my ($this) = @_;
    
    $this->Root(new IMPL::DOM::Document(nodeName => 'Root')) or failed "Failed to create a document";
};

test InsertNode => sub {
    my ($this) = @_;
    my $child = $this->Root->insertNode(new IMPL::DOM::Node(nodeName => 'Child')) or failed "Failed to insert a child node";
    failed "fiestChild returned incorrect results" unless ($this->Root->firstChild || 0) == $child;
};

test AppendNode => sub {
    my ($this) = @_;
    
    my $child = $this->Root->appendNode(new IMPL::DOM::Node(nodeName => 'Child')) or failed "Failed to append a child node";
    
    my $lastChild = $this->Root->removeLast;
    
    failed "removeLast returned incorrect results" unless $lastChild == $child;
};

test GetDocumentNode => sub {
    my ($this) = @_;
    
    my $child = $this->Root->firstChild->appendNode(new IMPL::DOM::Node(nodeName => 'GrandChild')) or failed "Failed to append a child node";
    
    failed "document property is undef" unless $child->document;
    failed "document property returned incorrect value" unless $child->document == $this->Root;
};

test DocumentCreateNode => sub {
	my ($this) = @_;
	
	my $child = $this->Root->firstChild->appendNode($this->Root->Create(Info => { uuid => '77f9-9a-6d58' } )) or failed "Failed to append a child node";
    
    failed "document property is undef" unless $child->document;
    failed "document property returned incorrect value" unless $child->document == $this->Root;
};

test MoveNode => sub {
    my ($this) = @_;
    
    my $grandChild = $this->Root->firstChild->firstChild;
    $this->Root->appendNode($grandChild);
    
    failed "incorrect new parentNode value" unless ($grandChild->parentNode || 0) == $this->Root;
    failed "incorrect new document value" unless ($grandChild->document || 0) == $this->Root;
};

test AppendRange => sub {
    my ($this) = @_;
    
    my $count = $this->Root->childNodes->Count;
    
    $this->Root->appendRange(
        map IMPL::DOM::Node->new(nodeName => "Item", nodeValue => $_),1..10
    );
    
    failed
        "Wrong number of a child nodes",
        "Expected: ".($count+10),
        "Actual: ".$this->Root->childNodes->Count
    unless $count + 10 == $this->Root->childNodes->Count;
};

test SelectNodes => sub {
    my ($this) = @_;
    
    my @result = $this->Root->selectNodes("Item");
    
    failed
        "Wrong number of a selected nodes",
        "Expected: 10",
        "Actual: ".scalar(@result)
    unless @result == 10;
};

test SelectNodesByQuery => sub {
    my ($this) = @_;
    
    my @result = $this->Root->selectNodes(sub { $_->nodeName =~ /child/i } );
    failed
        "Wrong number of a selected nodes",
        "Expected: 2",
        "Actual: ".scalar(@result)
    unless @result == 2;
};

test SelectNodesPath => sub {
	my ($this) = @_;
	
	my @result = $this->Root->selectNodes('Child','Info');
	
	failed "Failed to select a node by path 'Child/Info'" unless @result;
};

test SelectByAxisDescendant => sub {
	my ($this) = @_;
	
	my @result = $this->Root->selectNodes( { descendant => ['GrandChild','Info']} );
	
	failed "Failed to select a node by path '//(GrandChild|Info)/'" unless @result == 2;
};

test SelectByAxisAncestor => sub {
	my ($this) = @_;
	
	my @result = $this->Root->selectSingleNode( { descendant => 'Info'} )->selectNodes( { ancestor => undef } ) ;
	
	failed "Failed to select a node by path '//Info/ancestor:*'" unless @result == 2;
};

test CheckNodesValues => sub {
    my ($this) = @_;
    
    my @expected = (1..10);
    
    my @result = map $_->nodeValue, grep $_->nodeValue, $this->Root->selectNodes("Item");
    
    failed
        "Some nodes returned wrong node values or in a wrong order",
        "Expected: ".join(', ',@expected),
        "Recieved: ".join(', ',@result)
    unless cmparray(\@expected,\@result);
    
    failed
        "a text property of a root node returned a wrong value",
        "Expected: @expected",
        "Recieved: ". $this->Root->text
    unless $this->Root->text eq join '',@expected;
};

test isComplex => sub {
    my ($this) = @_;
    
    failed "property isComplex returned false for the root node" unless $this->Root->isComplex;
    failed "property isComplex returned true for a simple node", $this->Root->selectSingleNode('Item')->childNodes->Count if $this->Root->selectSingleNode('Item')->isComplex;
};

test setObjectProperty => sub {
	my ($this) = @_;
	
	my $node = Test::DOM::TypedNode->new(nodeName => 'TestNode');
	
	my $name = 'Vergon 6';
	
	$node->nodeProperty(name => $name);
	failed "Failed to set a property 'name'", "Expected: $name", "Got: ".$node->name unless $node->name eq $name;
	
	$name = 'entity_vergon_6';
	$node->systemName($name);
	failed "Failed to set a property 'systemName'", "Expected: $name", "Got: ".$node->nodeProperty('systemName') unless $node->nodeProperty('systemName') eq $name;
};

test setDynamicProperty => sub {
	my $node = Test::DOM::TypedNode->new(nodeName => 'TestNode');
	
	my $uuid = 'entity_76fd98b9e7a';
	$node->nodeProperty(uuid => $uuid);
	failed "Failed to set a dynamic property 'uuid'", "Expected: $uuid", "Got: ".$node->nodeProperty('uuid') unless $node->nodeProperty('uuid') eq $uuid;
};

test setPrivateProperty => sub {
	my $node = Test::DOM::TypedNode->new(nodeName => 'TestNode');
	
	eval {
		$node->nodeProperty(_private => 'failed');
		1;
	} and failed "Setting a private property successfull";
};

test createNodeWithProps => sub {
	my $uuid = 'entity_76fd98b9e7a';
	my $name = 'Vergon 6';
	my $systemName = 'entity_vergon_6';
	
	my $node = Test::DOM::TypedNode->new(
		nodeName => 'TestNode',
		uuid => $uuid,
		name => $name,
		systemName => $systemName 
	);
	
	failed "Failed to get dynamic property 'uuid'" unless $node->nodeProperty('uuid') eq $uuid;
	failed "Failed to get property 'name' through nodeProperty method" unless $node->nodeProperty('name') eq $name;
	failed "Failed to get property name directly" unless $node->name eq $name;
};

test listNodePredefinedProps => sub {
	my $node = Test::DOM::TypedNode->new(nodeName => 'TestNode');
	
	my @props = $node->listProperties;
	my @expected = qw(name _private);
	
	failed "Got wrong list of props", @props unless cmparray(\@props,\@expected);
};

test listNodeAllProps => sub {
	my $node = Test::DOM::TypedNode->new(
		nodeName => 'TestNode',
		uuid => 'ade58f98b', # dynamic
		name => 'noname', # predefined
		systemName => 'no sys' # not visible to DOM
	);
	
	my @props = $node->listProperties;
	my @expected = qw(name _private uuid); # systemName is not a DOM prop
	
	failed "Got wrong list of props", @props unless cmparray(\@props,\@expected);
};

test MemoryLeaks => sub {
	my $doc = new IMPL::DOM::Document(nodeName => 'Root');
	weaken($doc);
	
	assert(not defined $doc);
};

package Test::DOM::TypedNode;
use parent qw(IMPL::DOM::Node);
use IMPL::Class::Property;
use IMPL::DOM::Property qw(_dom);

__PACKAGE__->PassThroughArgs;

BEGIN {
	public _dom property name => prop_all;
	public property systemName => prop_all;
	private _dom property _private => prop_all;   
}


1;
