package IMPL::Web::Application::RestResource;
use strict;

use IMPL::lang qw(:declare :constants is);
use IMPL::Exception();

use IMPL::declare {
	require => {
		ForbiddenException => 'IMPL::Web::ForbiddenException',
		NotFoundException => 'IMPL::Web::NotFoundException',
		InvalidOpException => '-IMPL::InvalidOperationException',
		ArgumentException => '-IMPL::InvalidArgumentException',
		TTransform => '-IMPL::Transform',
		TResolve => '-IMPL::Config::Resolve',
		CustomResource => 'IMPL::Web::Application::CustomResource'
	},
	base => {
		'IMPL::Web::Application::RestCustomResource' => '@_'
	}
};

BEGIN {
	public property target => PROP_GET | PROP_OWNERSET;
	public property index => PROP_GET | PROP_OWNERSET;
	public property fetch => PROP_GET | PROP_OWNERSET;
	
	public property methods => PROP_GET | PROP_OWNERSET;
	
	public property childRegex => PROP_GET | PROP_OWNERSET;
	public property enableForms => PROP_GET | PROP_OWNERSET;
	
}

sub CTOR {
	my ($this) = @_;
	
	die ArgumentException->new("target") unless $this->target;
	
	$this->final($this->childRegex ? 0 : 1);
	
	if ($this->enableForms) {
        $this->methods->{create} = {
        	get => \&_ParentGet,
        	post => \&_ParentPost,
        	final => 1 # this resource doesn't have any children
        };
        
        $this->methods->{edit} = {
        	get => \&_ParentGet,
        	post => \&_ParentPut,
        	final => 1 # this resource doesn't have any children
        };
        
        $this->methods->{delete} {
        	get => \&_ParentGet,
        	post => \&_ParentDelete
        	final => 1 # this resource doesn't have any children
        }
	}
}

sub _ParentGet {
	my ($this,$action) = @_;
	return $this->parent->GetImpl($action);
}

sub _ParentPut {
    my ($this,$action) = @_;
    return $this->parent->PutImpl($action);
}

sub _ParentPost {
    my ($this,$action) = @_;
    return $this->parent->PostImpl($action);
}

sub _ParentDelete {
    my ($this,$action) = @_;
    return $this->parent->DeleteImpl($action);
}

sub FetchChildResource {
	my ($this,$id,$action) = @_;
	
	my $rx = $this->childRegex;
	
	my $res;
	
	if (length $id == 0) {
		
		my $method = $this->index;
		die ForbiddenException->new() unless $method;
		
		$res = $this->InvokeMember($method,$action);
		
	} elsif (my $resource = $this->methods->{$id}) {
		return CustomResource->new(
            get => $resource->{get},
            post => $resource->{post},
            put => $resource->{put},
            delete => $resource->{delete},
            parent => $this,
            id => $id,
            target => $this->target
		);
		 
	} elsif ($rx and $id =~ m/^$rx$/ and $method = $this->fetch) {
		$res = $this->InvokeMember($method,$action, { id => $id } );
	} else {
		die ForbiddenException->new();
	}
	
    die NotFoundException->new() unless defined $res;
        
    return $this->contract->Transform($res, {parent => $this, id => $id} );
}

1;

__END__

=pod

=head1 NAME

C<IMPL::Web::Application::RestResource> - ресурс Rest вебсервиса.

=head1 SYNOPSIS

=begin text

[REQUEST]
GET /artists

[RESPONSE]
<artists>
    <artist id="1">
        <name>The Beatles <name/>
    </atrist>
    <artist id="2">
        <name>Bonobo</name>
    </artist>
</artists>

[REQUEST]
GET /artists/1/cds?title='Live at BBC'

[RESPONSE]
<cds>
    <cd id="14">
        <title>Live at BBC 1</title>
    </cd>
    <cd id="15">
        <title>Live at BBC 2</title>
    </cd>
</cds>

[REQUEST]
GET /cds/15

[RESPONSE]
<cd id="15">
    <title>Live at BBC 2</title>
</cd>

=end text

=begin code

use IMPL::require {
	TRes => 'IMPL::Web:Application::RestResource',
	DataContext => 'My::App::DataContext'
};

my $cds = TRes->new(
    DataContext->Default,
    {
    	methods => {
    		history => {
    			get => {
	    			method => 'GetHistory',
	    			parameters => [qw(from to)]
    			}, 
    		},
    		rating => {
    			get => {
    				method => 'GetRating'
    			}
    			post => {
    				method => 'Vote',
    				parameters => [qw(id rating comment)]
    			}
    		}
    	}
    	list => 'search',
    	fetch => 'GetItemById'
    }   
);

=end code

=head1 DESCRIPTION

Каждый ресурс представляет собой коллекцию и реализует методы C<HTTP> C<GET,POST,PUT,DELETE>.

Ресурсы выстраиваются в иерархию, на основе пути. Поиск конечного реурса происходит последовательным
вызовом метода GET с именем очередного ресурса. 
  

=head2 HTTP METHODS

=head3 C<GET>

Возвращает коллекцию дочерних ресурсов.

=head3 C<GET {id}>

Возвращает дочерний объект с идентификатором C<id>

=head3 C<GET {method}>

Вызывает метод C<method> и возвращает его результаты. При публикации методов доступных
через C<GET> данные методы не должны вносить изменений в предметную область.

=head3 C<PUT {id}>

Обновляет дочерний ресурс с указанным идентификатором.

=head3 C<DELETE {id}>

Удаляет дочерний ресурс с указанным идентификатором.

=head3 C<POST>

Добавляет новый дочерний ресурс в коллекцию.

=head3 C<POST {method}>

Вызывает метод C<method>.

=head2 HTTP METHOD MAPPING 

=head3 C<POST {method}>

Вызывает метод C<method>, в отличии от C<GET> методы опубликованные через C<POST> могут вносить
изменения в объекты. 

=head1 BROWSER COMPATIBILITY

Однако существует проблема с браузерами, поскольку тег C<< <form> >> реализет только методы
C<GET,POST>. Для решения данной проблемы используется режим совместимости C<enableForms>. В
случае когда данный режим активен, автоматически публикуются дочерние ресурсы C<create,edit,delete>.

=head2 C<GET create>

Возвращает C<target>.

=head2 C<POST create>

Вызывает метод C<PostImpl> передавая ему свои параметры.

=head2 C<GET edit>

Возвращает C<target>.

=head2 C<POST edit>

Вызывает метод C<$this->parent->PutImpl($this->id)> передавая ему свои параметры.

=head2 C<GET delete>.

Возвращает C<target>.

=head2 C<POST delete>.

Вызывает метод C<$this->parent->DeleteImpl($this->id)> передавая ему свои параметры.

=head1 MEMBERS

=head2 C<[get]id>

Идентификатор текущего ресурса.

=head2 C<[get]target>

Объект (также может быть и класс), обеспечивающий функционал ресурса.

=head2 C<[get]parent>

Родительский ресурс, в котором находится текущий ресурс. Может быть C<undef>,
если текущий ресурс является корнем.

=head2 C<[get]methods>

Содержит описания методов, которые будут публиковаться как дочерние ресурсы.

=head2 C<[get]childRegex>

Содержит регулярное выражение для идентификаторов дочерних объектов. Если оно
не задано, то данный ресурс не является коллекцией.

=head2 C<[get]fetch>

Содержит описание метода для получения дочернего объекта. Если данный метод
отсутствует, то дочерние ресурсы не получится адресовать относительно данного.
По умолчанию получает идентификатор дочернего ресурса первым параметром.  

=head2 C<[get]list>

Описание метода для получения списка дочерних объектов. По умолчанию не
получает параметров.

=head2 C<[get]insert>

Описание метода для добавление дочернего ресурса. По умолчанию получает
объект C<CGI> описывабщий текущий запрос первым параметром.

=head2 C<[get]update>

Описание метода для обновления дочернего ресурса. По умолчанию получает
идентификатор дочернего ресурса и объект C<CGI> текущего запроса.

=head2 C<[get]delete>

Описание метода для удаления дочернего ресурса. По умолчанию получает
идентификатор дочернего ресурса.

=head2 C<GetImpl($child,$action)>

=over

=item C<$child>

Идентификатор дочернего ресутсра

=item C<$action>

Текущий запрос C<IMPL::Web::Application::Action>.

=back

Переадресует запрос нужному методу внутреннего объекта C<target> при
помощи C<InvokeMember>.

=head2 C<PutImpl($child,$action)>

=over

=item C<$child>

Идентификатор дочернего ресутсра

=item C<$action>

Текущий запрос C<IMPL::Web::Application::Action>.

=back

Переадресует запрос нужному методу внутреннего объекта C<target> при
помощи C<InvokeMember>.

=head2 C<PostImpl($child,$action)>

=over

=item C<$child>

Идентификатор дочернего ресутсра

=item C<$action>

Текущий запрос C<IMPL::Web::Application::Action>.

=back

Переадресует запрос нужному методу внутреннего объекта C<target> при
помощи C<InvokeMember>.

=head2 C<DeleteImpl($child,$action)>

=over

=item C<$child>

Идентификатор дочернего ресутсра

=item C<$action>

Текущий запрос C<IMPL::Web::Application::Action>.

=back

Переадресует запрос нужному методу внутреннего объекта C<target> при
помощи C<InvokeMember>.

=head2 C<InvokeMember($memberInfo,$child,$action)>

=over

=item C<$memberInfo>

Описание члена внутреннего объекта C<target>, который нужно вызвать.

=item C<$child>

Идентификатор дочернего ресутсра

=item C<$action>

Текущий запрос C<IMPL::Web::Application::Action>.

=back

Вызывает метод внутреннего объекта C<target>, предварительно подготовив
параметры на основе описания C<$memberInfo> и при помощи С<MakeParameter()>.

=head2 C<MakeParameter($paramDef,$child,$action)>

=over

=item C<$paramDef>

Описание параметра, может быть C<IMPL::Transform> или простая строка.

Если описание параметра - простая строка, то ее имя либо

=over

=item C<id>

Идентификатор дочернего ресурса

=item C<query>

Объект C<CGI> текущего запроса

=item C<action>

Текущий запрос C<IMPL::Web::Application::Action>

=item C<любое другое значение>

Интерпретируется как параметр текущего запроса.

=back

Если описание параметра - объект C<IMPL::Transform>, то будет выполнено это преобразование над C<CGI>
объектом текущего запроса C<< $paramDef->Transform($action->query) >>.

=back

=cut