package IMPL::Web::Security;
use strict;

use IMPL::Security::Auth qw(:Const);
use IMPL::declare {
    require => {
        Exception => 'IMPL::Exception',
        NotImplementedException => '-IMPL::NotImplementedException',
        ArgException => '-IMPL::InvalidArgumentException',
        SecurityContext => 'IMPL::Security::AbstractContext'
    },
};

use constant {
    ERR_NO_SUCH_USER => -1,
    ERR_NO_SEC_DATA => -2,
    ERR_NO_AUTHORITY => -3,
    ERR_NO_SEC_CONTEXT => -4,
    ERR_AUTH_FAIL => -5
};

sub AuthUser {
    my ($this,$name,$package,$challenge) = @_;
    
    my $user = $this->users->GetById($name)
        or return {
            status => AUTH_FAIL,
            code => ERR_NO_SUCH_USER
        };

    my $auth;
    if ( my $secData = $user->GetSecData($package) ) {
        $auth = $package->new($secData);
    } else {
        return {
            status => AUTH_FAIL,
            code => ERR_NO_SEC_DATA,
            user => $user
        };
    }
    
    return {
    	status => AUTH_FAIL,
    	code => ERR_NO_SEC_CONTEXT
    } unless SecurityContext->current;
    
    return {
    	status => AUTH_FAIL,
    	code => ERR_NO_AUTHORITY
    } unless SecurityContext->current->authority;

    my $status = SecurityContext->current->authority->InitSession(
        $user,
        [$user->roles],
        $auth,
        $challenge
    );
    
    return {
        status => $status,
        code => ($status == AUTH_FAIL ? ERR_AUTH_FAIL : 0),
        user => $user
    };
}

sub Logout {
	my ($this) = @_;
	
	my $session = SecurityContext->current;
	if($session && $session->authority) {
		$session->authority->CloseSession($session);
		
		$this->sessions->Delete($session);
	}
}

sub CreateSecData {
    my ($this,$package,$params) = @_;
    
    die ArgException->new(params => 'A hash reference is required')
        unless ref($params) eq 'HASH';
    
    return $package->CreateSecData(%$params);
}

1;

__END__

=pod

=head1 NAME

C<IMPL::Web::Security> Модуль для аутентификации и авторизации веб запроса.

=head1 DESCRIPTION

Текущий модуль обеспечивает функции верхнего уровня для работы с системой
безопасности. Поскольку модуль является абстрактым, конкретные функции
хранения и реализацию объектов модели безопасности должно обеспечить само
приложение.

Сама система безопасности в веб приложении состоит из двух частей

=over

=item Модель системы безопасности

Предоставляет такие объкты безопасности, как пользователь, сессия роль,
определяет правила проверки прав доступа субъекта к объекту.

=item Модуль безопасности

Контекст безопасности создается именно этим модулем.

Как правило встраивается в транспортный уровеь в виде обработчика
C<IMPL::Web::Handler> и реализует непосредственно протокол аутентификации и
обмена с пользователем.

Также модуль безопасности использует модель для хранения сессий и данных
аутентификции. 

=back

=head1 MEMBERS

=head2 C<AuthUser($name,$package,$challenge)>

Инициирует создание новой сессии используя провайдера безопасности текущего
контекста безопасности.

=over

=item C<$name>

Имя пользователя, которое будет использоваться при поиске его в БД. 

=item C<$package>

Имя модуля аутентификации, например, C<IMPL::Security::Auth::Simple>.

=item C<$challenge>

Данные, полученные от клиента, которые будут переданы модулю аутентификации для
начала процесса аутентификации и создания сессии.

=back

Функция возвращает хеш с элементами

=over

=item C<status>

Статус аутентификации - отражает общее состояние процесса ацтентификации,

=over

=item C<AUTH_FAIL>

Аутентификация неудачная, сессия не создана.

=item C<AUTH_INCOMPLETE>

Аутентификация требует дополнительных шагов, сессия создана, но еще не доверена.

=item C<AUTH_SUCCESS>

Аутентификация успешно проведена, сессия создана.

=back

=item C<code>

=back

=cut
