package IMPL::DOM::Schema::ValidationError;
use strict;
use warnings;

use overload
    '""' => \&toString,
    'fallback' => 1;

use parent qw(IMPL::Object);
use IMPL::Class::Property;
use IMPL::Class::Property::Direct;
use IMPL::Resources::Format qw(FormatMessage);

BEGIN {
    public _direct property Node => prop_get; # target document node (if exists)
    public _direct property Schema => prop_get; # a schema for the target node (if exists) 
    public _direct property Source => prop_get; # a schema which triggered this error (can be equal to the Schema)
    public _direct property Parent => prop_get; 
    public _direct property Message => prop_get; # displayable message
}

sub CTOR {
    my ($this,%args) = @_;
    
    $this->{$Node} = $args{Node};
    $this->{$Schema} = $args{Schema} if $args{Schema};
    $this->{$Source} = $args{Source} if $args{Source};
    if ($args{Parent}) {
        $this->{$Parent} = $args{Parent};
    } elsif ($args{Node}) {
        $this->{$Parent} = $args{Node}->parentNode;
    } else {
        die new IMPL::InvalidArgumentException("A 'Parent' or a 'Node' parameter is required");
    }
    $this->{$Message} = FormatMessage(delete $args{Message}, \%args) if $args{Message};
}

sub toString {
    (my $this) = @_;
    return $this->Message;
}

1;

__END__

=pod

=head1 NAME

C<IMPL::DOM::Schema::ValidationError> - Описывает ошибку в документе.

=head1 DESCRIPTION

При проверке документа на ошибки формирования возвращается массив с объектами
C<IMPL::DOM::Schema::ValidationError>, каждая из которых описывает одну ошибку
в документе.

С помощью данного объекта осущетсвляется привязка элемента схемы, элемента документа
и сообщения о причине возникновения ошибки.

=head1 MEMBERS

=over

=item C<[get] Node>

Узел в документе который привел к ошибке. Как правило это либо простые узлы, либо
узлы, которые не могут присутствоать в данном месте по схеме.

Данное свойство может быть C<undef>. 

=item C<[get] Parent>

Родительский узел в котором произошла ошибка. Используется в случаях, когда C<Node>
не указан, например, если по схеме должен существовать дочерний узел с определенным
именем, а в реальном документе его нет.

Также это свойство может использоваться при формировании сообщения.

=item C<[get] Schema>

Схема для C<Node> или узла который должен присутсвовать если C<Node> не задан.

=item C<[get] Source>

Схема, проверка которой привела к возникновению ошибки. Поскольку схемы могут
использовать ссылки, то данное свойство нужно для получения схемы узла, а не
схемы его типа.

Тоесть проверка схемы узла C<IMPL::DOM::Schema::Node> приводит к проверке схемы
типа, например, C<IMPL::DOM::Schema::ComplexType>, а свойство C<Source> будет
указывать именно на C<IMPL::DOM::Schema::Node>.

=item C<[get] Message>

Возвращает форматированное сообщение об ошибке.

=item C<toString()>

Преобразует ошибку к строке, возвращает значение свойства C<Message>

=back

=head1 REMARKS

=begin code

my $doc = IMPL::DOM::XMLReader->LoadDocument('data.xml');
my $schema = IMPL::DOM::Schema->LoadSchema('schema.xml');

my @errors = $schema->Validate($doc);

my $node = $doc->selectSingleNode('user','name');

# Получаем все ошибки относящиеся к данному узлу
my @nodeErrors = grep { ($_->Node || $_->Parent) == $node } @errors;  

=end code

=cut
