package IMPL::Web::Application;
use strict;
use warnings;

use base qw(IMPL::Config IMPL::Object::Singleton);

require IMPL::Web::Application::Action;
require IMPL::Web::Application::Response;

use IMPL::Class::Property;
use CGI;

__PACKAGE__->PassThroughArgs;

BEGIN {
    public property handlerError => prop_all;
    public property actionFactory => prop_all;
    public property handlersQuery => prop_all | prop_list;
    public property responseCharset => prop_all;
    public property security => prop_all;
    public property options => prop_all;
    public property fetchRequestMethod => prop_all;
}

sub CTOR {
	my ($this) = @_;
	
	$this->actionFactory('IMPL::Web::Application::Action') unless $this->actionFactory;
	$this->responseCharset('utf-8') unless $this->responseCharset;
	$this->fetchRequestMethod(\&defaultFetchRequest) unless $this->fetchRequestMethod;
	$this->handlerError(\&defaultHandlerError) unless $this->handlerError;
}

sub Run {
    my ($this) = @_;
    
    while (my $query = $this->FetchRequest()) {
        
        my $action = $this->actionFactory->new(
        	query => $query,
        	application => $this, 
        );
        
        eval {
	        $action->response->charset($this->responseCharset);
	        
	        $action->ChainHandler($_) foreach $this->handlersQuery;
	        
	        $action->Invoke();
	        
	        $action->response->Complete;
        };
        if ($@) {
        	my $e = $@;
        	# we are expecting this method to be safe otherwise we can trust nothing in this wolrd 
        	$this->handlerError()->($this,$action,$e);
        }
    }
}

sub FetchRequest {
	my ($this) = @_;
	
	if( ref $this->fetchRequestMethod eq 'CODE' ) {
		return $this->fetchRequestMethod->($this);
	} else {
		die new IMPL::Exception("Unknown fetchRequestMethod type",ref $this->fetchRequestMethod);
	}
}

{
	my $hasFetched = 0;

	sub defaultFetchRequest {
		my ($this) = @_;
		return undef if $hasFetched;
		$hasFetched = 1;
		my $query = CGIWrapper->new();
		$query->charset($this->responseCharset);
		return $query;
	}
}

sub defaultHandlerError {
	my ($this,$action,$e) = @_;
	warn $e;
	if ( eval {	$action->ReinitResponse(); 1; } ) {
		$action->response->contentType('text/plain');
		$action->response->charset($this->responseCharset);
		$action->response->status(500);
		my $hout = $action->response->streamBody;
		print $hout $e;
		$action->response->Complete();
	}	
}

package CGIWrapper;
use base qw(CGI);

use Encode;

sub param {
	my $this = shift;
	
	if (wantarray) {
		my @result = $this->SUPER::param(@_);
		
		return map Encode::is_utf8($_) ? $_ : Encode::decode($this->charset,$_,Encode::LEAVE_SRC), @result;
	} else {
		my $result = $this->SUPER::param(@_);
		
		return Encode::is_utf8($result) ? $result : Encode::decode($this->charset,$result,Encode::LEAVE_SRC);
	}

}

1;

__END__

=pod

=head1 SYNOPSIS

=begin code

require MyApp;

my $instance = spawn MyApp('app.config');

$instance->Run();

=end code

=head1 DESCRIPTION

C< use base qw( IMPL::Config IMPL::Object::Singleton )>

      ,   ,
     CGI    .

      

=over

=item 1

 cgi 

=item 2

  C<IMPL::Web::Application::Action>

=item 3

     C<< IMPL::Web::Application::Action->ChainHandler >>

=item 4

  C<< IMPL::Web::Application::Action->Invoke >>

=cut

     ,    
 .      C< IMPL::Configuration >.  
    C<options>,      
   , .   C<CONFIGURATION>. 

=head2 CONFIGURATION

    

=begin code xml

<?xml version="1.0" encoding="UTF-8"?>
<Application id='app' type="Test::Web::Application::Instance">
	
	<!-- Begin custom properties -->
	<name>Sample application</name>
	<dataSource type='IMPL::Config::Activator' id='ds'>
		<factory>IMPL::Object</factory>
		<parameters type='HASH'>
			<db>data</db>
			<user>nobody</user>
		</parameters>
	</dataSource>
	<securityMod type='IMPL::Config::Activator'>
		<factory>IMPL::Object</factory>
		<parameters type='HASH'>
			<ds refid='ds'/>
		</parameters>
	</securityMod>	
	<!-- End custom properties -->
	
	<!-- direct access to the activators -->
	<options type="HASH">
		<dataSource refid='ds'/>
	</options>
	
	<!-- Set default output encoding, can be changed due query handling -->
	<responseCharset>utf-8</responseCharset>
	
	<!-- Actions creation configuration -->
	<actionFactory type="IMPL::Object::Factory">
		
		<!-- Construct actions -->		
		<factory>IMPL::Web::Application::Action</factory>
		<parameters type='HASH'>
			
			<!-- with special responseFactory -->
			<responseFactory type='IMPL::Object::Factory'>
			
				<!-- Where resopnses have a special streamOut -->
				<factory>IMPL::Web::Application::Response</factory>
				<parameters type='HASH'>
				
					<!-- in memory dummy output instead of STDOUT -->
					<streamOut>memory</streamOut>
					
				</parameters>
			</responseFactory>
		</parameters>
	</actionFactory>
	
	<!-- Query processing chain -->
	<handlersQuery type="IMPL::Object::List">
		<item type="IMPL::Web::QueryHandler::PageFormat">
			<templatesCharset>cp1251</templatesCharset>
		</item>
	</handlersQuery>
</Application>

=end code xml

=head1 MEMBERS

=over

=item C<[get,set] handlerError>

        
   .     .

=item C<[get,set] actionFactory>

 ,   ,   
 C<IMPL::Web::Application::Action>   C<CGI> .

=begin code

my $action = $this->actionFactory->new(
    query => $query,
    application => $this, 
);

=end code

=item C< [get,set] fetchRequestMethod >

  CGI .  C<CGI>   , 
  ,   C<undef>. -  C<defaultFetchRequest>.

     ,     C<IMPL::Web::Application::RequestFetcher>.

=item C< [get,set,list] handlersQuery >

  ,     -.

=item C< [get,set] responseCharset>

  .

=item C< [get,set] security >

 C<IMPL::Web::Security>,     .

=item C< [get,set] options >

      ,   
  , ..     
    .
 
=back

=cut