package IMPL::DOM::Transform::PostToDOM;
use strict;
use warnings;

use IMPL::DOM::Navigator::Builder;
use IMPL::Class::Property;

use parent qw(IMPL::Transform);

BEGIN {
    public property documentClass => prop_get | owner_set;
    public property documentSchema => prop_get | owner_set;
    public property prefix => prop_get | owner_set;
    private property _navi => prop_all;
    public property Errors => prop_all | prop_list;
    private property _schema => prop_all;
}

our %CTOR = (
    'IMPL::Transform' => sub {
    	-plain => \&TransformPlain,
        HASH => \&TransformContainer,
        CGI => \&TransformCGI,
        CGIWrapper => \&TransformCGI
    }
);

sub CTOR {
	my ($this,$docClass,$docSchema,$prefix) = @_;
	$docClass ||= 'IMPL::DOM::Document';
	
	$this->_navi(
		IMPL::DOM::Navigator::Builder->new(
			$docClass,
			$docSchema
		)
	);
	$this->_schema($docSchema);
	$this->prefix($prefix) if $prefix;
}

sub TransformContainer {
    my ($this,$data) = @_;
    
    my $navi = $this->_navi;
        
    foreach my $key (
    	sort { $a->[1] cmp $b->[1] || $a->[2] <=> $b->[2]}
    	map [$_,/(\w+)(?:\[(\d+)\])?/], keys %$data
    ){
    	my $value = $data->{$key->[0]};
    	my $node = $navi->NavigateCreate($key->[1]);
    	
    	$node->nodeProperty(instanceId => $key->[2]) if defined $key->[2];
    	
    	$this->Transform($value);
    	
    	$navi->Back();
    }
    
    return $navi->Current;
}

sub TransformPlain {
	my ($this,$data) = @_;
	
	$this->_navi->Current->nodeValue( $this->_navi->inflateValue($data) );
}

sub TransformCGI {
	my ($this,$query) = @_;

	my $data={};
	
	my $prefix = $this->prefix;
	
	foreach my $param (grep index($_,$prefix) >= 0 , $query->param()) {
		length (my $value = $query->param($param)) or next;
		
		my @parts = split /\//,$param;
		
		my $node = $data;
		while ( my $part = shift @parts ) {
			if (@parts) {
				$node = ($node->{$part} ||= {});
			} else {			
				$node->{$part} = $value;
			}
		}  
	}
	
	if (keys %$data > 1) {
		$data = { document => $data };
	}
	
	my $doc = $this->Transform($data);
	$doc->nodeProperty( query => $query );
	$this->Errors->Append( $this->_navi->BuildErrors);
	$this->Errors->Append( $this->_schema->Validate($doc));
	return $doc;
}

1;

__END__

=pod

=head1 NAME

C<IMPL::DOM::Transform::PostToDOM> -   C<CGI>  DOM .

=head1 SINOPSYS

=begin code

	my $schema = IMPL::DOM::Schema->LoadSchema('Data/user.add.schema.xml');
	
	my $transform = IMPL::DOM::Transform::PostToDOM->new(
		undef, # default class
		$schema,
		$schema->selectSingleNode('ComplexNode')->name
	);
	
	my $doc = $transform->Transform(
		CGI->new({
			'user/login' => 'bob',
			'user/fullName' => 'Bob Marley',
			'user/password' => 'secret',
			'user/password_retype' => 'secret',
			'user/birthday' => '1978-12-17',
			'user/email[1]' => 'bob@marley.com',
			'user/email[2]' => 'bob.marley@google.com',
			process => 1
		})
	);

=end code

=head1 DESCRIPTION

   CGI   DOM .     ,  
     C<prefix>.

    

=over

=item 1      ,        .

=item 2      ,    _

=item 3        ,    
  .
	
=item 4      '/'

=back 

=cut
