package IMPL::Security::Context;
use strict;
use warnings;

use parent qw(IMPL::Object IMPL::Object::Autofill);

__PACKAGE__->PassThroughArgs;

use IMPL::Class::Property;

require IMPL::Security::Principal;

my $current;
my $nobody;

BEGIN {
    public property principal => prop_get;
    public property rolesAssigned => prop_all | prop_list;
    public property auth => prop_all;
    public property authority => prop_all;
}

sub CTOR {
	my ($this) = @_;
	
	die new IMPL::InvalidArgumentException("The parameter is required", 'principal') unless $this->principal;
}

sub Impersonate {
    my ($this,$code) = @_;
    
    my $old = $current;
    $current = $this;
    my $result;
    my $e;
    
    {
	    local $@;
	    eval {
	        $result = $code->();
	    };
	    $e = $@;
    }
    $current = $old;
    if($e) {
        die $e;
    } else {
        return $result;
    }
}

sub Apply {
	my ($this) = @_;
	
	$current = $this;
}

sub isTrusted {
	my ($this) = @_;
	
	if (my $auth = $this->auth) {
		return $auth->isTrusted;
	} else {
		return 0;
	}
}

sub nobody {
    my ($self) = @_;
    $nobody = $self->new(principal => IMPL::Security::Principal->nobody) unless $nobody;
    $nobody;
}

sub current {
	my ($self) = @_;
	
	$current = __PACKAGE__->nobody unless $current;
	$current;
}

sub Satisfy {
	my ($this,@roles) = @_;
	
	my $roleEffective = new IMPL::Security::Role ( _effective => scalar $this->rolesAssigned );
	
	return $roleEffective->Satisfy(@roles);
}

1;

__END__

=pod

=head1 NAME

C<IMPL::Security::Context> -  .

=head1 SINOPSYS

=begin code

my $context = IMPL::Security::Context->nobody;

my $result = $context->Impersonate(
	sub {
		# do some untrusted code
	}
);

=end code

=head1 DESCRIPTION

C<[Autofill]>

   ,    ,  
     ,    C<nobody>.

=head1 MEMBERS

=over

=item C<CTOR(%props)>

     .

=item C<[get] principal>

 ,  .

=item C<[get] rolesAssigned>

  ()  .

=item C<[get] auth>

  C<IMPL::Security::Auth>,     .

=item C<[static,get] authority>

  ,   .

=item C<[get] isTrusted>

     ,   .

=item C<Impersonate($code)>

          C<$code>.  
,  .

=item C<Apply()>

    ,      C<Impersonate>, 
  .

=item C<[static,get] nobody>

   ,  .

=item C<[static,get] current>

 .

=back

=cut