package Test::DOM::Builder;
use strict;
use warnings;

use parent qw(IMPL::Test::Unit);
__PACKAGE__->PassThroughArgs;

use IMPL::Class::Property;
use IMPL::Test qw(test failed shared);

require IMPL::DOM::Schema;
require IMPL::DOM::Navigator::Builder;
require IMPL::DOM::Navigator::SimpleBuilder;
require IMPL::DOM::Document;
require IMPL::DOM::XMLReader;

BEGIN {
    public property schemaDoc => prop_all;
}

sub CTOR {
    my ($this) = @_;
    
    my $schema = new IMPL::DOM::Schema;
    $schema->appendRange(
        IMPL::DOM::Schema::ComplexNode->new( name => 'personInfo' )->appendRange(
            IMPL::DOM::Schema::NodeSet->new()->appendRange(
                new IMPL::DOM::Schema::SimpleNode( name => 'firstName' ),
                new IMPL::DOM::Schema::SimpleNode( name => 'lastName' ),
                new IMPL::DOM::Schema::ComplexNode( name => 'address', maxOccur => 'unbounded' )->appendRange(
                    IMPL::DOM::Schema::NodeSet->new()->appendRange(
                        new IMPL::DOM::Schema::SimpleNode( name => 'street' ),
                        new IMPL::DOM::Schema::SimpleNode( name => 'line', minOccur => 0 )
                    )
                )
            )
        )
    );
    
    $this->schemaDoc( $schema );
}

test CreateBuilder => sub {
    my ($this) = @_;
    
    my $builder = IMPL::DOM::Navigator::Builder->new(
        new IMPL::DOM::Document(nodeName => 'personInfo'),
        $this->schemaDoc
    );
};

test BuildDocument => sub {
    my ($this) = @_;
    
    my $builder = IMPL::DOM::Navigator::Builder->new(
        'IMPL::DOM::Document',
        $this->schemaDoc
    );
    
    $builder->NavigateCreate('personInfo', version => '1');
        $builder->NavigateCreate('firstName')->nodeValue('Nemo');
        $builder->Back();
        $builder->NavigateCreate('lastName')->nodeValue('Nobel');
        $builder->Back();
        for(1..10) {
            $builder->NavigateCreate('address', local => 1);
                $builder->NavigateCreate('street')->nodeValue('Hellroad');
                $builder->Back();
                $builder->NavigateCreate('line')->nodeValue($_);
                $builder->Back();
            $builder->Back();
        }
    $builder->Back();
    
    my @errors = $this->schemaDoc->Validate($builder->Document);
    failed ("The built document doesn't pass a validation",@errors) if @errors;
    
    return 1;
};

test BuildSimpleDocument => sub {
    my ($this) = @_;
    
    my $builder = IMPL::DOM::Navigator::SimpleBuilder->new();
    
    $builder->NavigateCreate('personInfo', version => '1');
        $builder->NavigateCreate('firstName')->nodeValue('Nemo');
        $builder->Back();
        $builder->NavigateCreate('lastName')->nodeValue('Nobel');
        $builder->Back();
        for(1..10) {
            $builder->NavigateCreate('address', local => 1);
                $builder->NavigateCreate('street')->nodeValue('Hellroad');
                $builder->Back();
                $builder->NavigateCreate('line')->nodeValue($_);
                $builder->Back();
            $builder->Back();
        }
    $builder->Back();
    
    my @errors = $this->schemaDoc->Validate($builder->Document);
    failed ("The built document doesn't pass a validation",@errors) if @errors;
    
    return 1;
};

test BuildDocumentFromXml => sub {
    my ($this) = @_;
    
    my $builder = IMPL::DOM::Navigator::SimpleBuilder->new();
    my $reader = IMPL::DOM::XMLReader->new( Navigator => $builder );
    
    $reader->ParseFile("Resources/person_info.xml");
    
    my $doc = $builder->Document() or failed("No document was constrcuted");
    
    my @errors = $this->schemaDoc->Validate($doc);
    failed("The document isn't correct", @errors) if @errors;
    my $name = ($doc->selectNodes("firstName"))[0]->nodeValue;
    failed("The firstName has a wrong value", "Expected: Norman", "Got: $name") unless $name eq "Norman";
    
};


1;
