package IMPL::Object::Singleton;
use strict;
use warnings;

require IMPL::Exception;
use parent qw(
    IMPL::Class::Meta
);

__PACKAGE__->static_accessor_own(_instance => undef);

my %instances;

sub CTOR {
    die IMPL::InvalidOperationException->new("Only one instance of the singleton can be created",ref $_[0])
        if $_[0]->_instance;
}

sub instance {
    my $this = shift;
    return $this->_instance || $this->_instance($this->Activate());
}

sub Activate {
    die IMPL::NotImplementedException->new("Activation isn't implemented", shift);
}

sub Release {
    shift->_instance(undef);
}

1;

__END__

=pod

=head1 SYNOPSIS

=begin code

package Foo;

use parent qw(IMPL::Object IMPL::Object::Singleton);

#....

Foo->isnatnce->some_work();

Foo->isnatnce->get_result();

=end code

=head1 DESCRIPTION

Реализует шаблон Singleton. Наследники данного класса могут иметь только один
экземпляр. Создать этот экземпляр можно явно, используюя конструктор, либо
автоматически при обращении к свойству C<instance>, для этого нужно
переопределить метод C<Activate()>

=head1 MEMBERS

=head2 C<CTOR()>

Проверяет на единственность экземпляра класса, запоминает созданный экземпляр.

=head2 C<[static,get]instance>

Текущий экземпляр класса, если он еще не создан, то вызывает метод C<Activate>.

=head2 C<[static,abstract]Activate()>

Вызывается автоматически при обращении к свойству C<instance>, если экземпляр
объекта еще не был создан.

=head2 C<[static]Release()>

Освобождает текущий экземпляр.

=cut
