package IMPL::Security::Auth::Simple;
use strict;

use parent qw(IMPL::Object IMPL::Security::Auth);
use Digest::MD5;

use IMPL::Class::Property;
use IMPL::Security::Auth qw(:Const);

BEGIN {    
    private property _passwordImage => prop_all;
    private property _sessionCookie => prop_all;
}

sub CTOR {
    my ($this,$secData) = @_;
    
    my ($passImg,$cookie) = split /\|/,$secData;
    
    $this->_passwordImage($passImg);
    $this->_sessionCookie($cookie);
}

sub secData {
    my ($this) = @_;
    
    if ($this->_sessionCookie) {
        return join ('|',$this->_passwordImage, $this->_sessionCookie );
    } else {
        return $this->_passwordImage;
    }
}

sub isTrusted {
    my ($this) = @_;
    
    $this->_sessionCookie ? 1 : 0;
}

sub DoAuth {
    my ($this,$challenge) = @_;

    if (Digest::MD5::md5_hex($challenge) eq $this->_passwordImage) {
        return (AUTH_SUCCESS,$this->_sessionCookie($this->GenSSID));
    } elsee {
        return (AUTH_FAIL,$this->_sessionCookie(undef));
    }
}

sub ValidateSession {
    my ($this,$cookie) = @_;
    
    die new IMPL::InvalidOperationException("The context is untrusted") unless $this->_sessionCookie;
    
    if ($cookie eq $this->_sessionCookie) {
        return (AUTH_SUCCESS,undef);
    } else {
        return (AUTH_FAIL,undef);
    }
}

sub CreateSecData {
    my ($self,%args) = @_;
    
    die new IMPL::InvalidArgumentException("The parameter is required",'password') unless $args{password};
    
    return Digest::MD5::md5_hex($args{password});
}

sub SecDataArgs {
    password => 'SCALAR'        
}

1;

__END__

=pod

=head1 NAME

C<IMPL::Security::Auth::Simple> Модуль простой авторизации.

=head1 DESCRIPTION

Использует алгоритм MD5 для хранения образа пароля.

=head1 MEMBERS

=over

=item C<CTOR($secData)>

Создает объект аутентификации, передавая ему данные для инициализации.

=item C<[get]secData>

Возвращает данные безопасности, которые можно использовать для восстановления
состояния объекта.

=item C<[get]isTrusted>

Является ли объект доверенным для аутентификации сессии (тоесть хранит данные
для аутентификации сессии).

=item C<DoAuth($challenge)>

Аутентифицирует пользователя. Используется один этап. C<$challenge>
открытый пароль пользователя.

Возвращает C<($status,$challenge)>

=over

=item C<$status>

Результат либо C<AUTH_SUCCESS>, либо C<AUTH_FAIL>

=item C<$challenge>

В случае успеха возвращает cookie (уникальный номер) сессии

=back

=item C<ValidateSession($challenge)>

Проверяет аутентичность сессии. Использует один этап. C<$challenge> cookie
сессии, полученный при выполнении метода C<DoAuth>.

Возвращает C<($status,$challenge)>

=over

=item C<$status>

Результат либо C<AUTH_SUCCESS>, либо C<AUTH_FAIL>

=item C<$challenge>

Всегда C<undef>

=back

=back

=cut
