package IMPL::Web::Application::ControllerUnit;

use base qw(IMPL::Object);

use IMPL::Class::Property;

BEGIN {
	public property action => prop_get | owner_set;
	public property application => prop_get | owner_set;
	public property query => prop_get | owner_set;
	public property formData => prop_get | owner_set;
	public property formSchema => prop_get | owner_set;
	public property formErrors => prop_get | owner_set;
}

sub CTOR {
	my ($this,$action) = @_;
	
	$this->action($action);
	$this->application($action->application);
	$this->query($action->query);
}

sub InvokeAction {
	my ($self,$method,$action) = @_;
	
	if ($self->can($method)) {
		my $unit = $self->new($action);
		$unit->$method();
	} else {
		die new IMPL::InvalidOperationException("Invalid method call",$self,$method);
	}
}

1;

__END__

=pod

=head1 NAME

C<IMPL::Web::Application::ControllerUnit> -        .

=head1 DESCRIPTION

,        ,  
  .        , 
     .

     ,      .
    C<InvokeAction($method,$action)>,  / 
.

        .

=head2  

    ,   ,    
 .

=head2 

     ,   DOM    .
  DOM   .      
C<formData>, C<formSchema>, C<formErrors>.

      ,     , 
  

=begin code

{
	state => '{ new | correct | invalid }',
	result => $transactionResult,
	formData => $formDOM,
	formSchema => $formSchema,
	formErrors => @errors
}

=end code

=over

=item C<state>

  .

=over

=item C<new>

  ,    ,   .
      .

=item C<correct>

  ,  ,       C<result>

=item C<invalid>

    ,     C<formErrors>. 
 .

=back

=item C<result>

  ,    .

=item C<formData>

    .   ,     ,
     ,   .

=item C<formSchema>

  ,      .

=item C<formErrors>

  ,   

=back

=head1 MEMBERS

=over

=item C<[get] application>

=item C<[get] query>

=item C<[get] response>

=item C<[get] formData>

=item C<[get] formSchema>

=item C<[get] formErrors>

=item C<InvokeAction($method,$action)>

   ,       
. 

=back

=head1 EXAMPLE

=begin code

package MyBooksUnit;
use strict;
use base qw(IMPL::Web::Application::ControllerUnit);

__PACKAGE__->PassThroughArgs;

__PACKAGE__->transactions(qw(
	find 
	info
));
__PACKAGE__->forms(
	create => 'books.create.xml'
);

sub find {
	my ($this) = @_;
	
	return $this->application->dataSource->resultset('Books')->find({author => $this->query->param('author')});
}

sub info {
	my ($this) = @_;
	
	return $this->application->dataSource->resultset('Books')->find({id => $this->query->param('id')});
}

sub create {
	my ($this) = @_;
	
	my %book = map {
		$_ => $this->formData->selectSingleNode($_)->nodeValue
	} qw(author_id title year ISBN);
	
	return $this->application->datasource->resultset('Books')->create(\%book);
}

=end code

=cut