package IMPL::Security::Auth::Simple;
use strict;

use Digest::MD5 qw(md5_hex);

use IMPL::Security::Auth qw(:Const);

use IMPL::Const qw(:prop);
use IMPL::declare {
    require => {
        Exception => 'IMPL::Exception',
        WrongDataException => '-IMPL::WrongDataException'        
    },
    base => [
        'IMPL::Security::Auth' => undef,
        'IMPL::Object' => undef
    ],
    props => [
        _stage => PROP_ALL,
        _salt => PROP_ALL,
        _image => PROP_ALL
    ]
};

use constant {
    STAGE_INIT => 1,
    STAGE_DONE => 2    
};

sub CTOR {
    my ($this,$secData) = @_;
    
    my ($stage,$salt,$img) = split /\|/,$secData;
    
    die WrongDataException->new()  unless grep $_ == $stage, (STAGE_INIT, STAGE_DONE);
    
    $this->_stage($stage);
    $this->_salt($salt);
    $this->_image($img);
    
}

sub secData {
    my ($this) = @_;
    
    return join ('|',$this->_stage, $this->_salt , $this->_image );
}

sub isTrusted {
    my ($this) = @_;
    
    $this->_stage == STAGE_DONE ? 1 : 0;
}

sub DoAuth {
    my ($this,$challenge) = @_;
    
    my $salt = $this->_salt;

    if (md5_hex($salt, $challenge, $salt) eq $this->_image) {
        if ($this->_stage == STAGE_INIT) {
            $this->_stage(STAGE_DONE);
        }
        return (AUTH_SUCCESS, undef);
    } elsee {
        return (AUTH_FAIL, undef);
    }
}

sub CreateSecData {
    my ($self,%args) = @_;
    
    die new IMPL::InvalidArgumentException("The parameter is required",'password') unless $args{password};
    
    my $salt = $self->GenSSID();
    return return join ('|',STAGE_INIT, $salt, md5_hex($salt,$args{password},$salt));
}

sub SecDataArgs {
    password => 'SCALAR'
}

1;

__END__

=pod

=head1 NAME

C<IMPL::Security::Auth::Simple> Модуль простой авторизации.

=head1 DESCRIPTION

Использует алгоритм MD5 для хранения образа пароля.

=head1 MEMBERS

=head2 C<CTOR($secData)>

Создает объект аутентификации, передавая ему данные для инициализации.

=head2 C<[get]secData>

Возвращает данные безопасности, которые можно использовать для восстановления
состояния объекта.

=head2 C<[get]isTrusted>

Является ли объект доверенным для аутентификации сессии (тоесть хранит данные
для аутентификации сессии).

=head2 C<DoAuth($challenge)>

Аутентифицирует пользователя. Используется один этап. C<$challenge>
открытый пароль пользователя или cookie сессии.

Возвращает C<($status,$challenge)>

=over

=item C<$status>

Результат либо C<AUTH_SUCCESS>, либо C<AUTH_FAIL>

=item C<$challenge>

В случае успеха возвращает cookie (уникальный номер) сессии, либо C<undef>

=back

=cut
