package IMPL::Security::Context;
use strict;
use warnings;

use IMPL::require {
    AbstractContext => 'IMPL::Security::AbstractContext',
};

use IMPL::declare {
    require => {
        Principal => 'IMPL::Security::Principal',
        Role => 'IMPL::Security::Role',
        Exception => 'IMPL::Exception',
        ArgumentException => '-IMPL::InvalidArgumentException'        
    },
    base => [
        'IMPL::Object' => undef,
        'IMPL::Object::Autofill' => '@_',
        'IMPL::Security::AbstractContext' => undef,
    ],
    props => [
        @{AbstractContext->abstractProps()}
    ]
};

__PACKAGE__->abstractProps([]);


my $nobody;

sub CTOR {
    my ($this) = @_;
    
    die ArgumentException->new("The parameter is required", 'principal') unless $this->principal;
}

sub nobody {
    my ($self) = @_;
    $nobody = $self->new(principal => Principal->nobody) unless $nobody;
    $nobody;
}

sub isTrusted {
    return 1;
}

1;

__END__

=pod

=head1 NAME

C<IMPL::Security::Context> - реализация контекста безопасности создаваемого в 
приложении.

=head1 SYNOPSIS

=begin code

my $context = IMPL::Security::Context->nobody;

my $result = $context->Impersonate(
    sub {
        # do some untrusted code
    }
);

$context = IMPL::Security::Context->new(
    principal => $user,
    assignedRoles => [
        $backupRole,
        $controlRole
    ]
);

$context->Impersonate(
    sub {
        
        # do some authorized operations
        
        $service->backupData('current.bak');
        $service->stop();
    }
);

=end code

=head1 DESCRIPTION

C<autofill>

Данная реализация контекста безопасности не привязана ни к источнику данных
ни к пакету аутентификации и авторизации, ее приложение может создать в любой
момент, при этом система сама несет ответственность за последствия.

Данный контекст нужен для выполнения системой служебных функций.

=head1 MEMBERS

см. также C<IMPL::Security::AbstractContext>.

=head2 C<CTOR(%props)>

Создает объект и заполняет его свойствами. C<principal> должен быть обязательно
указан.

=head2 C<[static,get] nobody>

Контекст для неаутентифицированных пользователей, минимум прав.

=cut
