package IMPL::Web::Application::Response;
use strict;

use base qw(IMPL::Object IMPL::Object::Autofill);

require IMPL::Exception;
require CGI;
require CGI::Cookie;

use Carp;
use Encode;
use IMPL::Class::Property;

#todo: add binary method to set a binary encoding, set it automatic when type isn't a text 

BEGIN {
	public property query => prop_get | owner_set; # cgi query
	public property status => prop_all, { validator => \&_checkHeaderPrinted };
	public property contentType => prop_all, { validator => \&_checkHeaderPrinted }; # String
	public property charset => { get => \&_charset, set => \&_charset }, { validator => \&_checkHeaderPrinted };
	public property expires => prop_all, { validator => \&_checkHeaderPrinted };
	public property cookies => prop_all, { validator => \&_checkHeaderPrinted }; # Hash
	
	public property buffered => prop_all, { validator => \&_canChangeBuffer }; # Boolean
	public property streamOut => prop_get | owner_set; # stream
	public property streamBody => {get => \&getStreamBody }; # stream
	public property isHeaderPrinted => prop_get | owner_set; # Boolean 
	
	private property _bufferBody => prop_all;
	private property _streamBody => prop_all;
}

__PACKAGE__->PassThroughArgs;

sub CTOR {
	my ($this,%args) = @_;
	
	$this->query(CGI->new($this->query() | {})) unless $this->query;
	
	$this->streamOut(*STDOUT) unless $this->streamOut;
	$this->buffered(1) unless defined $this->buffered;
	binmode $this->streamOut, ":encoding(".$this->charset.")";
}

sub _checkHeaderPrinted {
	my ($this,$value) = @_;
	
	die new IMPL::InvalidOperationException() if $this->isHeaderPrinted;
}

sub _canChangeBuffer {
	my ($this,$value) = @_;
	
	die new IMPL::InvalidOperationException() if $this->isHeaderPrinted or $this->_streamBody;
}

sub _charset {
	my $this = shift;
	
	if (@_) {
		my $charset = $this->query->charset(@_);
		
		my $hout = $this->streamOut;
		
		binmode $hout;
		binmode $hout, ":encoding($charset)";
		
		return $charset;
	} else {
		return $this->query->charset;
	}
}

sub _PrintHeader {
	my ($this) = @_;
	
	unless ($this->isHeaderPrinted) {
		$this->isHeaderPrinted(1);
		
		my %opt;
		
		$opt{-type} = $this->contentType if $this->contentType;
		$opt{-status} = $this->status if $this->status;
		$opt{-expires} = $this->expires if $this->expires;
		
		my $refCookies = $this->cookies;
		$opt{-cookie} = [map CGI::Cookie->new(-name => $_, $refCookies->{$_} ), keys %$refCookies] if $refCookies;
		
		my $hOut = $this->streamOut;
		
		print $hOut $this->query->header(
			%opt
		);
	}
}

sub getStreamBody {
	my ($this) = @_;
	
	return undef unless $this->streamOut;
	
	unless ($this->_streamBody) {
		if ($this->buffered) {
			my $buffer = "";
			
			$this->_bufferBody(\$buffer);
				
			open my $hBody, ">:encoding(utf-8)", \$buffer or die new IMPL::Exception("Failed to create buffer",$!);
			
			Encode::_utf8_on($buffer);
				
			$this->_streamBody($hBody);
		} else {
			$this->_PrintHeader();
			$this->_streamBody($this->streamOut);
		}
	}
		
	return $this->_streamBody;
}

sub Complete {
	my ($this) = @_;
	
	return 0 unless $this->streamOut;
	
	my $hOut = $this->streamOut;
	
	$this->_PrintHeader();
	
	$this->_streamBody(undef);
	
	if ($this->buffered) {
		print $hOut ${$this->_bufferBody};	
	}	
	
	$this->_bufferBody(undef);
	$this->streamOut(undef);
	
	return 1;
}

sub Discard {
	my ($this) = @_;
	
	carp "Discarding sent response" if $this->isHeaderPrinted;
	
	$this->_streamBody(undef);
	$this->_bufferBody(undef);
	$this->streamOut(undef);
}

1;

__END__

=pod

=head1 DESCRIPTION

   CGI ,        .

      ,     
   .

 C< isHeaderPrinted >        -
 . 

=head1 PROPERTIES

=head2 HTTP Header

    HTTP .         
   .      C< IMPL::InvalidOperationException >.

=over

=item C< query >

CGI ,     ,   .  .

=item C< status >

  HTTP. , '200 OK'.    ,      '200 '.

=item C< contentType >

 MIME.    ,  'text/html'.

=item C< charset >

,   query->charset.

=item C< expires >

   ,  '+10m'.       .

=item C< cookies >

   cookies,  C< { cart => ['foo','bar'], display => 'list' } >.

=back

=head2 Response behaviour

    .

=over

=item C< buffered >

C< True > -             C< Complete >,
       C< Complete >. 

C< False > -        ,   
       C< streamBody >

             .

=item C< streamOut >

  CGI .

=item C< streamBody >

     .

=item C< isHeadPrinted >

 ,      .

=back

=head1 METHODS

=over

=item C< Complete >

  .

=item C< Discard >

  ,      (, )
  ,    STDERR.

=back

=cut