package Test::DOM::Schema;
use strict;
use warnings;

use parent qw(IMPL::Test::Unit);
use IMPL::Test qw(test failed shared);
use IMPL::Class::Property;

__PACKAGE__->PassThroughArgs;

use IMPL::DOM::Schema;
use IMPL::DOM::Navigator::Builder;

BEGIN {
    shared public property SampleSchema => prop_all;
}

test GetMetaSchema => sub {
    my $metaSchema = IMPL::DOM::Schema->MetaSchema();
};

test AutoverifyMetaSchema => sub {
    my $metaSchema = IMPL::DOM::Schema->MetaSchema();

    if (my @errors = $metaSchema->Validate($metaSchema)) {
        failed "Self verification failed", map $_ ? $_->Message : 'unknown', @errors;
    }
};

test VerifyCorrectSchema => sub {
    my ($this) = @_;
    my $metaSchema = IMPL::DOM::Schema->MetaSchema();
    
    my $schema = new IMPL::DOM::Schema;
    $schema->appendRange(
        IMPL::DOM::Schema::ComplexNode->new( name => 'personInfo' )->appendRange(
            IMPL::DOM::Schema::NodeSet->new()->appendRange(
                new IMPL::DOM::Schema::SimpleNode( name => 'firstName' ),
                new IMPL::DOM::Schema::SimpleNode( name => 'lastName' ),
                new IMPL::DOM::Schema::ComplexNode( name => 'address' )->appendRange(
                    IMPL::DOM::Schema::NodeSet->new()->appendRange(
                        new IMPL::DOM::Schema::SimpleNode( name => 'street' ),
                        new IMPL::DOM::Schema::SimpleNode( name => 'line', minOccur => 0 )
                    )
                )
            )
        )
    );
    
    $this->SampleSchema($schema);
    
    my @errors = $metaSchema->Validate($schema);
    failed "Failed to validate a wellformed schema", map $_->Message, @errors if @errors;
};

test VerifyWrongSchema => sub {
    my $metaSchema = IMPL::DOM::Schema->MetaSchema();
    
    my $schema = new IMPL::DOM::Schema;
    $schema->appendRange(
        IMPL::DOM::Schema::ComplexNode->new( name => 'personInfo' )->appendRange(
            new IMPL::DOM::Schema::ComplexType( type => 'someType' ),
            new IMPL::DOM::Schema::SimpleNode( name => 'lastName' ),
            new IMPL::DOM::Schema::ComplexNode( name => 'address' )->appendRange(
                new IMPL::DOM::Schema::SimpleNode( name => 'street' ),
                new IMPL::DOM::Schema::SimpleNode( name => 'line' )
            )
        )
    );
    
    my @errors = $metaSchema->Validate($schema);
    failed "A not wellformed schema validated correctly" unless @errors;
};

test ValidateCorrectData => sub {
    my ($this) = @_;
    
    my $data = IMPL::DOM::Node->new(nodeName => 'personInfo')->appendRange(
        IMPL::DOM::Node->new(nodeName => 'firstName', nodeValue => 'John'),
        IMPL::DOM::Node->new(nodeName => 'lastName', nodeValue => 'Smith'),
        IMPL::DOM::Node->new(nodeName => 'address')->appendRange(
            IMPL::DOM::Node->new(nodeName => 'street', nodeValue => 'main road')
        )
    );
    
    if (my @errors = $this->SampleSchema->Validate($data)) {
        failed "Failed to validate a correct data", map $_->Message , @errors;
    }
};

test ValidateWrongData => sub {
    my ($this) = @_;
    
    my $data = IMPL::DOM::Node->new(nodeName => 'personInfo')->appendRange(
        IMPL::DOM::Node->new(nodeName => 'firstName', nodeValue => 'John'),
        IMPL::DOM::Node->new(nodeName => 'address')->appendRange(
            IMPL::DOM::Node->new(nodeName => 'street', nodeValue => 'main road')
        )
    );
    
    failed "A wrong data validated corretly" unless $this->SampleSchema->Validate($data);
};

test LoadXmlSchemaTypes => sub {
    my ($this) = @_;
    
    my $schema = IMPL::DOM::Schema->LoadSchema("Resources/types.xml") or failed "Failed to parse schema";
    
    return 1;
};

test LoadXmlSchemaData => sub {
    my ($this) = @_;
    
    my $schema = IMPL::DOM::Schema->LoadSchema("Resources/form.xml") or failed "Failed to parse schema";
    
    return 1;
};


1;
