package IMPL::Web::View::TTControl;
use strict;

use IMPL::Const qw(:prop);
use IMPL::lang qw(:hash);
use Scalar::Util qw(blessed);
use IMPL::declare {
	require => {
	    TemplateDocument => 'Template::Document',
        TTContext => 'Template::Context',
        Exception => 'IMPL::Exception',
        ArgumentException => '-IMPL::InvalidArgumentException',
        OperationException => '-IMPL::InvalidOperationException'
	},
	base => [
	   'IMPL::Object' => undef
	],
	props => [
	   id => PROP_RO,
	   attributes => PROP_RW,
	   name => PROP_RO,
	   context => PROP_RO,
	   template => PROP_RO
	]
};


{
    my $nextId = 1;
    sub _GetNextId {
        return $nextId++;
    }
}

our $AutoloadRegex = qr/^[a-z]/;
our @REFLECT_META = qw(title layout);

sub CTOR {
    my ($this,$name,$template,$context,$refProps) = @_;
    
    $name ||= "control";
    
    $this->template( $template ) or die new IMPL::ArgumentException("A template is required");
    $this->context( $context ) or die new IMPL::ArgumentException("A context is required");
    
    $this->id($name . "-" . _GetNextId()) unless $this->id;
    
    $this->name($name);
    $this->attributes({});
    
    my %attrs;
    
    foreach my $meta ( @REFLECT_META ) {
        next if $meta =~ /^_/;
        if( my $value = $template->$meta() ) {
            $attrs{$meta} = $value;
        }
    }
    
    hashApply(\%attrs,$refProps) if ref $refProps eq 'HASH';
    
    while (my($key,$value) = each %attrs) {
        $this->SetAttribute($key,$value);
    }
}

sub InitInstance {
    my ($this,$args) = @_;
    
    $args ||= {};
    
    if ( my $ctor = $this->template->blocks->{CTOR} ) {
        $this->context->include($ctor, { %$args, this => $this, template => $this->template } );
    }
}

sub GetAttribute {
    my ($this,$name) = (shift,shift);
    
    if (my $method = $this->can($name)) {
        unshift @_,$this;
        goto &$method;
    } else {
        return $this->attributes->{$name};
    }
}

sub SetAttribute {
    my $this = shift;
    my $name = shift;
    
    if (my $method = $this->can($name)) {
        unshift @_, $this;
        goto &$method;
    } else {
        return $this->attributes->{$name} = shift;
    }
}

sub GetMainBlock {
    $_[0]->template->blocks->{RENDER} || $_[0]->template;
}

sub Render {
    my ($this,$args) = @_;
    
    $args = {} unless ref $args eq 'HASH';
    
    if(my $body = $this->GetMainBlock ) {
        return $this->context->include( $body, { %$args, this => $this, template => $this->template } );
    } else {
        return "";
    }    
}

sub RenderBlock {
    my ($this, $block, $args) = @_;
    
    $args = {} unless ref $args eq 'HASH';
    
    return $block ? $this->context->include( $block, { %$args, this => $this, template => $this->template } ) : undef;
}

sub ExportBlock {
    my ($this,$block) = @_;
    
    return TemplateDocument->new({
        BLOCK => $this->template->blocks->{$block},
        DEFBLOCKS => $this->template->blocks
    });
}

sub AUTOLOAD {
    our $AUTOLOAD;
    
    my $method = ($AUTOLOAD =~ m/(\w+)$/)[0];
    
    return if $method eq 'DESTROY';
    
    if ($method =~ /$AutoloadRegex/) {
        my $this = shift;
        
        die OperationException->new("can't invoke method '$method' on an unblessed reference") unless blessed $this;
    
        return @_ ? $this->SetAttribute($method,@_) : $this->GetAttribute($method);
    } else {
        die OperationException->new("The specified method '$method' doesn't exists");
    }
}

1;

__END__

=pod

=head1 NAME

C<IMPL::Web::View::TTControl>

=head1 SYNPOSIS

=begin text

[%
    META version = 1;
    BLOCK INIT;
        # this is a document scope
        dojo.require.push( 'dijit/form/Input' );
    END;
    BLOCK CTOR;
        # local to this block
        TPreview = require('My/Org/TextPreview');
        
        # init control props 
        this.dojoClass = this.dojoClass || 'dijit.form.Input';
        this.visualClass = this.visualClass || 'classic';
        this.childNodes = [];
        
        # init content
        FOREACH text IN this.data;
            this.childNodes.push( TPreview.new('preview', nodeValue = text ) );
        END;
        
    END;
%]

<div class="$this.visualClass" data-dojo-type="$this.dojoClass">
    [% FOREACH node IN this.childNodes %]
        [% node.Render() %]
        <hr />
    [% END %]
</div>

=end text

=head1 DESCRIPTION

Представляет собой фрагмент документа, который имеет шаблон для отображения,
набор свойств и т.п.

=head2 BLOCKS

=head3 META

Атрибуты C<META> C<layout>, C<title> будут перенесены в свойства элемента
управления. 

=head3 INIT

Данный блок шаблона управления выполняется один раз при создании первого
экземпляра элемента управления, в пространстве имен документа. Может
использоваться для формирования заголовочной части документа, скрипта
подключающего C<js> модули и т.п.

Выполнение данного блока производится фабрикой элементов управления.

=head3 CTOR

данный блок выполняется каждый раз при создании нового экземпляра элемента
управления, при этом переменная C<this> указывает на эземпляр элемента
упарвления. Данный блок можно использовать для инициализации свойств элемента
управления.

=head3 RENDER

Данный блок выполняется при вызове метода C<Render()>, вывод данного блока и
есть результат отображения элемента управления. Если в шаблоне нет блока
C<RENDER>, то сам шаблон считается таковым. 

=head2 TEMPLATE VARS

Каждый шаблон имеет собственное пространство имен, вложенное в пространство имен
фабрики элементов (которая разделяет пространство имен документа). В шаблоне
могут определяться новые переменные, однако они останутся локальными для блоков.

Чтобы передать данные между блоками следует использовать ссылку на элемент
управления C<this>.

=begin text

[%
    BLOCK CTOR;
        this.extraCssClass = 'active';
        text = "this text will gone";
    END;
%]

<div class="$this.extraCssClass">some text $text</div>

=end text

В примере выше переменная C<$text> установленная в конструкторе шаблона, при
отображении элемента управления будет неопределена.

=over

=item * C<this>

ссылка на объект элемента управления

=item * C<component>

ссылка на текущий шаблон, устанавливается автоматически в методе
C<Template::Context::process>.

=item * C<template>

ссылка на шаблон элемента управления, для совместимости с C<TT>

=back

=head1 MEMBERS

=over

=item * C<[get]context>

Контекст элемента управления, хранит пременные шаблона. Передается в
конструкторе. Фабрика элементов управления создает новый контекст пространство
имен которого вложено в пространство имен документа.

=item * C<[get,set]template>

C<Template::Document> Шаблон элемента управления.

=item * C<AUTOLOAD>

Для удобства работы с шаблоном, элементы управления предоставляю доступ к своим
свойствам через метод C<AUTOLOAD>. Имена свойств должны начинаться со строчной
буквы.

=back

=cut