package IMPL::Config;
use strict;
use warnings;

use base qw(IMPL::Object::Accessor IMPL::Object::Serializable IMPL::Object::Autofill);

__PACKAGE__->PassThroughArgs;

use IMPL::Class::Member;
use IMPL::Class::PropertyInfo;
use IMPL::Exception;

use IMPL::Serialization;
use IMPL::Serialization::XmlFormatter;



sub LoadXMLFile {
    my ($self,$file) = @_;
    
    my $class = ref $self || $self;
    
    my $serializer = new IMPL::Serializer(
        Formatter => new IMPL::Serialization::XmlFormatter(
            IdentOutput => 1,
            SkipWhitespace => 1
        )
    );
    
    open my $hFile,'<',$file or die new IMPL::Exception("Failed to open file",$file,$!);
    
    my $obj;
    eval {
        $obj = $serializer->Deserialize($hFile);
    };
    
    if ($@) {
        my $e=$@;
        die new IMPL::Exception("Can't load the configuration file",$file,$e);
    }
    return $obj;
}

sub SaveXMLFile {
    my ($this,$file) = @_;
    
    my $serializer = new IMPL::Serializer(
        Formatter => new IMPL::Serialization::XmlFormatter(
            IdentOutput => 1,
            SkipWhitespace => 1
        )
    );
    
    open my $hFile,'>',$file or die new IMPL::Exception("Failed to open file",$file,$!);
    
    $serializer->Serialize($hFile, $this);
}

sub xml {
    my $this = shift;
    my $serializer = new IMPL::Serializer(
        Formatter => new IMPL::Serialization::XmlFormatter(
            IdentOutput => 1,
            SkipWhitespace => 1
        )
    );
    my $str = '';
    open my $hFile,'>',\$str or die new IMPL::Exception("Failed to open stream",$!);
    
    $serializer->Serialize($hFile, $this);
    
    undef $hFile;
    
    return $str;
}

sub save {
    my ($this,$ctx) = @_;

    foreach my $info ($this->get_meta('IMPL::Class::PropertyInfo')) {
        next if $info->Access != IMPL::Class::Member::MOD_PUBLIC; # save only public properties

        my $name = $info->Name;
        $ctx->AddVar($name => $this->rawGet($name)) if $this->rawGet($name);
    }
    
}

sub spawn {
	goto &LoadXMLFile;
}

sub get {
	my $this = shift;
	
	if (@_ == 1) {
		my $obj = $this->SUPER::get(@_);
		return UNIVERSAL::isa($obj,'IMPL::Config::Activator') ? $obj->activate : $obj;
	} else {
		my @objs = $this->SUPER::get(@_);	
		return map UNIVERSAL::isa($_,'IMPL::Config::Activator') ? $_->activate : $_, @objs ;	
	}
}

sub rawGet {
	my $this = shift;
	return $this->SUPER::get(@_);
}

sub Exists {
	$_[0]->SUPER::get($_[1]) ? 1 : 0;
}

1;
__END__

=pod

=h1 SYNOPSIS

package App::Config
use base qw(IMPL::Config)

use IMPL::Class::Property;
use IMPL::Config::Class;

BEGIN {
    public property SimpleString => prop_all;
    public property MyClass => prop_all;
    public property DataSource => prop_all; 
}

sub CTOR {
    my $this = shift;

    $this->MyClass(new IMPL::Config::Class(Type => 'MyClass'')) unless $this->MyClass;
    $this->DataSource(
    	new IMPL::Config::Activator(
    		type => 'MyDataSource',
    		args=>{
    			host => 'localhost',
    			user => 'dbuser'
    		}
    	)
    ) unless $this->Exists('DataSource');
}

# in some script

my $app = spawn App::Config('default.xml');

$app->Run();

=head1 DESCRIPTION

 / .    
    ,   Public  
   .

=head1 MEMBERS

=over

=item C<< IMPL::Config->LoadXMLFile($fileName) >>

  XML   

=item C<< $instance->SaveXMLFile($fileName) >>

   

=item C<< xml >>

    XML 

=item C<< IMPL::Config->spawn($file) >>

  C<LoadXMLFile>

=back

=cut
