package Test::Web::View;
use strict;
use warnings;
use utf8;

use parent qw(IMPL::Test::Unit);
__PACKAGE__->PassThroughArgs;

use File::Slurp;

use IMPL::Test qw(assert test);
use IMPL::Web::View::TTLoader();

use constant {
	TTLoader => typeof IMPL::Web::View::TTLoader
};

sub templatesDir {
	$_[0]->GetResourceDir('Resources','TTView');
}

test TTLoaderTests => sub {
	my ($this) = @_;
	
	my $loader = TTLoader->new(
		{
			INCLUDE_PATH => [
				$this->templatesDir
			]
		},
		ext => '.tt',
		initializer => 'global.tt'
	);
	
	# test the loader to be able to find a desired resource
	assert( defined($loader->template('simple') ) );
	
	# loader should be initialized on demand
	assert( not $loader->isInitialized );
	
	# loader should be able to load a document
	my $doc = $loader->document('simple');
	assert(defined $doc);
	
	assert( $loader->isInitialized );
	assert( $loader->context->stash->get('user') eq 'test_user');
	
	# document should inherit loader's context 
	assert( $doc->context->stash->get('user') eq 'test_user');
};

test TTDocumentTests => sub {
	my ($this) = @_;
	my $loader = TTLoader->new(
		{
			INCLUDE_PATH => [
				$this->templatesDir
			],
			INTERPOLATE => 1,
			POST_CHOMP => 1,
			ENCODING => 'utf-8'
		},
		ext => '.tt',
		initializer => 'global.tt'
	);
	
	my $doc = $loader->document('simple');
	
	assert(defined $doc);
	
	assert($doc->nodeName eq 'document');
	assert(not $doc->can('notexists')); # autoloaded property should be ignored
	assert($doc->notexists eq ''); # nonexisting property 
	assert($doc->version == 10); # static metadata
	assert($doc->user eq 'test_user'); # global data
	assert($doc->templateVar eq 'initialized by the constructor'); # defined in CTOR block
	
	my $text = $doc->Render();
	my $expected = read_file($this->GetResourceFile('Resources','TTView.Output','simple.txt'), binmode => ':utf8');
	
	assert($text eq $expected, "Bad Render() output","Got: $text", "Expected: $expected");
	
};

test TTControlTests => sub {
	my ($this) = @_;
	
	my $loader = TTLoader->new(
		{
			INCLUDE_PATH => [
				$this->templatesDir
			],
			INTERPOLATE => 1,
			POST_CHOMP => 1,
			ENCODING => 'utf8'
		},
		ext => '.tt',
		initializer => 'global.tt'
	);
	
	my $doc = $loader->document('simple');
	
	assert(defined $doc);
	
	my $factory = $doc->require('My/Org/Panel');
	
	assert(defined $factory);
	
	assert($factory == $doc->require('My/Org/Panel'), "Control should be loaded only once");
	
	my $ctl = $factory->new('information', { visualClass => 'simple' } );
	
	assert(defined $ctl);
	
	
	assert($ctl->nodeName eq 'information', "Created control should have a name", "Got: ".$ctl->nodeName, "Expected: information");
	
	assert($ctl->nodeProperty('visualClass') eq 'simple');
	
	assert($ctl->controlObject == $ctl);
	
	assert($factory->instances == 1);
	
	assert($doc->context->stash->get('My.Org.Panel') == $factory);
	
	my $text = $ctl->Render();
	my $expected = read_file($this->GetResourceFile('Resources', 'TTView.Output', 'Panel.txt'), binmode => ':utf8');
	
	assert($text eq $expected, '$ctl->Render(): Bad output', "Got:      $text", "Expected: $expected");
	
};

1;