use strict;
package DOM::Gallery;
use Common;
our @ISA = qw(Object);

BEGIN {
    DeclareProperty(Id => ACCESS_READ);
    DeclareProperty(Name => ACCESS_READ);
    DeclareProperty(Description => ACCESS_READ);
    DeclareProperty(Images => ACCESS_READ);
    DeclareProperty(CurrentImage => ACCESS_READ);
    DeclareProperty(NextImage => ACCESS_READ);
    DeclareProperty(PrevImage => ACCESS_READ);
}

sub CTOR {
    my ($this,%args) = @_;
    
    $this->{$Id} = $args{'Id'};
    $this->{$Name} = $args{'Name'};
    $this->{$Description} = $args{'Description'};
}

sub GroupList {
    my ($this,$GroupCount, $option) = @_;
    
    my @images = map { $this->{$Images}->{$_} } sort keys %{$this->{$Images}};
    
    my @listGroups;
    my $group;
    for (my $i = 0; $i < $GroupCount; $i++ ) {
        #last unless scalar(@images) or $option =~ /align/i; 
        push (@$group, shift(@images));
        if ($i == $GroupCount - 1) {
            push @listGroups, $group;
            undef $group;
            $i = -1;
            last if not scalar(@images);
        }
    }
    
    return \@listGroups;
}

sub SelectImage {
    my ($this,$imageId) = @_;
    
    my @images = sort keys %{$this->{$Images}};
    
    for (my $i=0; $i <= @images; $i++) {
        if ($images[$i] eq $imageId) {
            $this->{$CurrentImage} = $this->{$Images}->{$images[$i]};
            $this->{$PrevImage} = $i-1 >= 0 ? $this->{$Images}->{$images[$i-1]} : undef;
            $this->{$NextImage} = $i+1 < @images ? $this->{$Images}->{$images[$i+1]} : undef;
            return 1;
        }
    }
    die new Exception("An image '$imageId' not found in the gallery '$this->{$Id}'");
}

sub AddImage {
    my ($this,$image) = @_;
    
    $this->{$Images}->{$image->Id()} = $image;
}

package DOM::Gallery::Image;
use Common;
our @ISA = qw(Object);
BEGIN {
    DeclareProperty(Id => ACCESS_READ);
    DeclareProperty(Name => ACCESS_READ);
    DeclareProperty(Gallery => ACCESS_READ);
    DeclareProperty(URL => ACCESS_READ);
    DeclareProperty(ThumbURL => ACCESS_READ);
}

sub CTOR {
    my ($this,%args) = @_;
    
    $this->{$Id} = $args{'Id'} or die new Exception ('An Id should be specified for an image');
    $this->{$Name} = $args{'Name'};
    $this->{$Gallery} = $args{'Gallery'} or die new Exception('An Gallery should be specified for an image');
    $this->{$URL} = $args{'URL'};
    $this->{$ThumbURL} = $args{'ThumbURL'};
}

package DOM::Providers::Gallery;
use Common;
our @ISA = qw(Object);

our $RepoPath;
our $ImagesURL;
our $Encoding;

BEGIN {
    DeclareProperty(GalleryCache => ACCESS_NONE);
    DeclareProperty(Repository => ACCESS_NONE);
}

sub CTOR {
    my ($this,%args) = @_;
    
    $this->{$Repository} = $args {'Repository'} or die new Exception('A path to an galleries repository should be specified');
}

sub GetProviderInfo() {
    return {
        Name => 'Gallery',
        Host => 'DOM::Site',
        Methods => {
            LoadGallery => \&SiteLoadGallery #($this,$site,$galleryId)
        }
    };
}

sub SiteLoadGallery {
    my ($this,$site,$galleryId) = @_;
    
    my $gallery = $this->LoadGallery($galleryId);
    
    $site->RegisterObject('Gallery',$gallery);
    
    return $gallery;
}

sub LoadGallery {
    my ($this,$galleryId) = @_;
    
    die new Exception("Invalid Gallery Id: $galleryId") if $galleryId =~ /\\|\//;
    
    my $galleryIdPath = $galleryId;
    $galleryIdPath =~ s/\./\//g;
    
    my $GalleryPath = $this->{$Repository} . $galleryIdPath .'/';
    
    die new Exception("A gallery '$galleryId' isn't found",$GalleryPath) if not -d $GalleryPath;
    
    open my $hDesc, "<:encoding($Encoding)", $GalleryPath.'index.htm' or die new Exception("Invalid gallery: $galleryId","Failed to open ${GalleryPath}index.htm: $!");
    
    my $GalleryName;   
    while (<$hDesc>) {
        if (/<title>(.+?)<\/title>/i) {
            $GalleryName = $1;
            last;
        }
    }    
    undef $hDesc;
    
    my $ImagesPath = $GalleryPath.'images/';
    my $ThumbsPath = $GalleryPath.'thumbnails/';
    
    opendir my $hImages, $ImagesPath or die new Exception("Invalid gallery: $galleryId","Can't open images repository: $!");
    
    my @imageIds = grep { -f $ImagesPath.$_ } readdir $hImages;
    
    my %imageNames;
    
    if (-f $GalleryPath.'description.txt') {
        local $/="\n";
        if (open my $hfile,"<:encoding($Encoding)",$GalleryPath.'description.txt') {
            while (<$hfile>) {
                chomp;
                my ($id,$name) = split /\s*=\s*/;
                $imageNames{$id} = $name;
            }
        }
    }
    
    undef $hImages;
    
    if ($Common::Debug) {
        foreach (@imageIds) {
            warn "A tumb isn't found for an image: $_" if not -f $ThumbsPath.$_;
        }
    }
    
    my $gallery = new DOM::Gallery(Id => $galleryId, Name => $GalleryName);
    
    foreach my $imageId (@imageIds) {
        $gallery->AddImage(new DOM::Gallery::Image(
                Id => $imageId,
                URL => $ImagesURL.$galleryIdPath.'/images/'.$imageId,
                ThumbURL => $ImagesURL.$galleryIdPath.'/thumbnails/'.$imageId,
                Gallery => $gallery,
                Name => $imageNames{$imageId}
            )
        );
    }
    
    return $gallery;
}

sub construct {
    my $self = shift;
    
    return new DOM::Providers::Gallery( Repository => $RepoPath);
}

1;
