package Test::DOM::Node;
use strict;
use warnings;

use base qw(IMPL::Test::Unit);
use IMPL::Test qw(test shared failed cmparray);
use IMPL::Class::Property;

require IMPL::DOM::Node;

__PACKAGE__->PassThroughArgs;

BEGIN {
    shared public property Root => prop_all;
}

test Create => sub {
    my ($this) = @_;
    
    $this->Root(new IMPL::DOM::Document(nodeName => 'Root')) or failed "Failed to create a document";
};

test InsertNode => sub {
    my ($this) = @_;
    my $child = $this->Root->insertNode(new IMPL::DOM::Node(nodeName => 'Child')) or failed "Failed to insert a child node";
    failed "fiestChild returned incorrect results" unless ($this->Root->firstChild || 0) == $child;
};

test AppendNode => sub {
    my ($this) = @_;
    
    my $child = $this->Root->appendNode(new IMPL::DOM::Node(nodeName => 'Child')) or failed "Failed to append a child node";
    
    my $lastChild = $this->Root->removeLast;
    
    failed "removeLast returned incorrect results" unless $lastChild == $child;
};

test GetDocumentNode => sub {
    my ($this) = @_;
    
    my $child = $this->Root->firstChild->appendNode(new IMPL::DOM::Node(nodeName => 'GrandChild')) or failed "Failed to append a child node";
    
    failed "document property is undef" unless $child->document;
    failed "document property returned incorrect value" unless $child->document == $this->Root;
};

test MoveNode => sub {
    my ($this) = @_;
    
    my $grandChild = $this->Root->firstChild->firstChild;
    $this->Root->appendNode($grandChild);
    
    failed "incorrect new parentNode value" unless ($grandChild->parentNode || 0) == $this->Root;
    failed "incorrect new document value" unless ($grandChild->document || 0) == $this->Root;
};

test AppendRange => sub {
    my ($this) = @_;
    
    my $count = $this->Root->childNodes->Count;
    
    $this->Root->appendRange(
        map IMPL::DOM::Node->new(nodeName => "Item", nodeValue => $_),1..10
    );
    
    failed
        "Wrong number of a child nodes",
        "Expected: ".($count+10),
        "Actual: ".$this->Root->childNodes->Count
    unless $count + 10 == $this->Root->childNodes->Count;
};

test SelectNodes => sub {
    my ($this) = @_;
    
    my @result = $this->Root->selectNodes("Item");
    
    failed
        "Wrong number of a selected nodes",
        "Expected: 10",
        "Actual: ".scalar(@result)
    unless @result == 10;
};

test SelectNodesByQuery => sub {
    my ($this) = @_;
    
    my @result = $this->Root->selectNodes(sub { $_->nodeName =~ /child/i } );
    failed
        "Wrong number of a selected nodes",
        "Expected: 2",
        "Actual: ".scalar(@result)
    unless @result == 2;
};

test CheckNodesValues => sub {
    my ($this) = @_;
    
    my @expected = (1..10);
    
    my @result = map $_->nodeValue, grep $_->nodeValue, $this->Root->selectNodes("Item");
    
    failed
        "Some nodes returned wrong node values or in a wrong order",
        "Expected: ".join(', ',@expected),
        "Recieved: ".join(', ',@result)
    unless cmparray(\@expected,\@result);
    
    failed
        "a text property of a root node returned a wrong value",
        "Expected: @expected",
        "Recieved: ". $this->Root->text
    unless $this->Root->text eq join '',@expected;
};

test isComplex => sub {
    my ($this) = @_;
    
    failed "property isComplex returned false for the root node" unless $this->Root->isComplex;
    failed "property isComplex returned true for a simple node", $this->Root->firstChild->nodeName if $this->Root->firstChild->isComplex;
};

package Test::DOM::TypedNode;
use base qw(IMPL::DOM::Node);
use IMPL::Class::Property;
use IMPL::DOM::Property;



1;
