package IMPL::DOM::Schema;
use strict;
use warnings;

require IMPL::DOM::Schema::ComplexNode;
require IMPL::DOM::Schema::ComplexType;
require IMPL::DOM::Schema::SimpleNode;
require IMPL::DOM::Schema::SimpleType;
require IMPL::DOM::Schema::Node;
require IMPL::DOM::Schema::AnyNode;
require IMPL::DOM::Schema::NodeList;
require IMPL::DOM::Schema::NodeSet;
require IMPL::DOM::Schema::Property;
require IMPL::DOM::Schema::SwitchNode;
require IMPL::DOM::Schema::Validator;
require IMPL::DOM::Navigator::Builder;
require IMPL::DOM::XMLReader;
require IMPL::DOM::Schema::InflateFactory;

use base qw(IMPL::DOM::Document);
use IMPL::Class::Property;
use IMPL::Class::Property::Direct;
use File::Spec;

our %CTOR = (
    'IMPL::DOM::Document' => sub { nodeName => 'schema' }
);

BEGIN {
    private _direct property _TypesMap => prop_all;
    public _direct property baseDir => prop_all;
    public _direct property BaseSchemas => prop_get | owner_set;
}

sub resolveType {
    $_[0]->{$_TypesMap}->{$_[1]};
}

sub CTOR {
	my ($this,%args) = @_;
	
	$this->{$baseDir} = ($args{baseDir} || '.');
}

sub Create {
    my ($this,$nodeName,$class,$refArgs) = @_;
    
    die new IMPL::Exception('Invalid node class') unless $class->isa('IMPL::DOM::Node');
    
    if ($class->isa('IMPL::DOM::Schema::Validator')) {
    	$class = "IMPL::DOM::Schema::Validator::$nodeName";
    	unless (eval {$class->can('new')}) {
    		eval "require $class; 1;";
    		my $e = $@;
			die new IMPL::Exception("Invalid validator",$class,$e) if $e;
    	}
    }
    
    return $this->SUPER::Create($nodeName,$class,$refArgs);
}

sub Process {
    my ($this) = @_;
    
    # process instructions
    $this->Include($_) foreach map $_->nodeProperty('source'), $this->selectNodes('Include');
    
    # build types map
    $this->{$_TypesMap} = { map { $_->type, $_ } $this->selectNodes(sub { $_[0]->nodeName eq 'ComplexType' || $_[0]->nodeName eq 'SimpleType' } ) };
}

sub Include {
	my ($this,$file) = @_;
	
	my $schema = $this->LoadSchema(File::Spec->catfile($this->baseDir, $file));
	
	$this->appendRange( $schema->childNodes );
}

sub LoadSchema {
	my ($this,$file) = @_;
	
	$file = File::Spec->rel2abs($file);
	
	my $class = ref $this || $this;
	
	my $reader = new IMPL::DOM::XMLReader(
		Navigator => new IMPL::DOM::Navigator::Builder(
			$class,
			$class->MetaSchema
		)
	);
		
	$reader->ParseFile($file);
	
	my $schema = $reader->Navigator->Document;
	
	my ($vol,$dir) = File::Spec->splitpath($file);
	
	$schema->baseDir($dir);
	
	my @errors = $class->MetaSchema->Validate($schema);
	
	die new IMPL::Exception("Schema is invalid",$file,map( $_->Message, @errors ) ) if @errors;
	
	$schema->Process;
	
	return $schema;
}

sub Validate {
    my ($this,$node) = @_;
    
    if ( my ($schemaNode) = $this->selectNodes(sub { $_->isa('IMPL::DOM::Schema::Node') and $_[0]->name eq $node->nodeName })) {
        $schemaNode->Validate($node);
    } else {
        return new IMPL::DOM::Schema::ValidationError(Node => $node, Message=> "A specified document (%Node.nodeName%) doesn't match the schema");
    }
}

my $schema;

sub MetaSchema {
    
    return $schema if $schema;
    
    $schema = new IMPL::DOM::Schema;
    
    $schema->appendRange(
        IMPL::DOM::Schema::ComplexNode->new(name => 'schema')->appendRange(
            IMPL::DOM::Schema::NodeSet->new()->appendRange(
                IMPL::DOM::Schema::Node->new(name => 'ComplexNode', type => 'ComplexNode', minOccur => 0, maxOccur=>'unbounded'),
                IMPL::DOM::Schema::Node->new(name => 'ComplexType', type => 'ComplexType', minOccur => 0, maxOccur=>'unbounded'),
                IMPL::DOM::Schema::Node->new(name => 'SimpleNode', type => 'SimpleNode', minOccur => 0, maxOccur=>'unbounded'),
                IMPL::DOM::Schema::Node->new(name => 'SimpleType', type => 'SimpleType', minOccur => 0, maxOccur=>'unbounded'),
                IMPL::DOM::Schema::Node->new(name => 'Node', type => 'Node', minOccur => 0, maxOccur=>'unbounded'),
                IMPL::DOM::Schema::SimpleNode->new(name => 'Include', minOccur => 0, maxOccur=>'unbounded')->appendRange(
                    IMPL::DOM::Schema::Property->new(name => 'source')
                )
            ),
        ),
        IMPL::DOM::Schema::ComplexType->new(type => 'NodeSet', nativeType => 'IMPL::DOM::Schema::NodeSet')->appendRange(
            IMPL::DOM::Schema::NodeSet->new()->appendRange(
                IMPL::DOM::Schema::Node->new(name => 'ComplexNode', type => 'ComplexNode', minOccur => 0, maxOccur=>'unbounded'),
                IMPL::DOM::Schema::Node->new(name => 'SimpleNode', type => 'SimpleNode', minOccur => 0, maxOccur=>'unbounded'),
                IMPL::DOM::Schema::Node->new(name => 'Node', type=>'Node', minOccur => 0, maxOccur=>'unbounded'),
                IMPL::DOM::Schema::SwitchNode->new(minOccur => 0, maxOccur => 1)->appendRange(
                    IMPL::DOM::Schema::Node->new(name => 'AnyNode', type => 'AnyNode'),
                    IMPL::DOM::Schema::Node->new(name => 'SwitchNode',type => 'SwitchNode')
                )
            )
        ),
        IMPL::DOM::Schema::ComplexType->new(type => 'SwitchNode', nativeType => 'IMPL::DOM::Schema::SwitchNode')->appendRange(
            IMPL::DOM::Schema::NodeSet->new()->appendRange(
                IMPL::DOM::Schema::Node->new(name => 'ComplexNode', type=>'ComplexNode', minOccur => 0, maxOccur=>'unbounded'),
                IMPL::DOM::Schema::Node->new(name => 'SimpleNode', type=>'SimpleNode', minOccur => 0, maxOccur=>'unbounded'),
                IMPL::DOM::Schema::Node->new(name => 'Node', type=>'Node', minOccur => 0, maxOccur=>'unbounded'),
            )
        ),
        IMPL::DOM::Schema::ComplexType->new(type => 'NodeList', nativeType => 'IMPL::DOM::Schema::NodeList')->appendRange(
            IMPL::DOM::Schema::NodeSet->new()->appendRange(
                IMPL::DOM::Schema::Node->new(name => 'ComplexNode', type => 'ComplexNode', minOccur => 0, maxOccur=>'unbounded'),
                IMPL::DOM::Schema::Node->new(name => 'SimpleNode', type => 'SimpleNode', minOccur => 0, maxOccur=>'unbounded'),
                IMPL::DOM::Schema::Node->new(name => 'SwitchNode',type => 'SwitchNode', minOccur => 0, maxOccur=>'unbounded'),
                IMPL::DOM::Schema::Node->new(name => 'Node', type => 'Node', minOccur => 0, maxOccur=>'unbounded'),
                IMPL::DOM::Schema::Node->new(name => 'AnyNode', type => 'AnyNode', minOccur => 0, maxOccur=>'unbounded'),
            )
        ),
        IMPL::DOM::Schema::ComplexType->new(type => 'ComplexType', nativeType => 'IMPL::DOM::Schema::ComplexType')->appendRange(
            IMPL::DOM::Schema::NodeList->new()->appendRange(
                IMPL::DOM::Schema::SwitchNode->new()->appendRange(
                    IMPL::DOM::Schema::Node->new(name => 'NodeSet', type => 'NodeSet'),
                    IMPL::DOM::Schema::Node->new(name => 'NodeList',type => 'NodeList'),
                ),
                IMPL::DOM::Schema::Node->new(name => 'Property', type=>'Property', maxOccur=>'unbounded', minOccur=>0),
                IMPL::DOM::Schema::AnyNode->new(maxOccur => 'unbounded', minOccur => 0, type=>'Validator')
            ),
            new IMPL::DOM::Schema::Property(name => 'type')
        ),
        IMPL::DOM::Schema::ComplexType->new(type => 'ComplexNode', nativeType => 'IMPL::DOM::Schema::ComplexNode')->appendRange(
            IMPL::DOM::Schema::NodeList->new()->appendRange(
                IMPL::DOM::Schema::SwitchNode->new()->appendRange(
                    IMPL::DOM::Schema::Node->new(name => 'NodeSet', type => 'NodeSet'),
                    IMPL::DOM::Schema::Node->new(name => 'NodeList',type => 'NodeList'),
                ),
                IMPL::DOM::Schema::Node->new(name => 'Property', type=>'Property', maxOccur=>'unbounded', minOccur=>0),
                IMPL::DOM::Schema::AnyNode->new(maxOccur => 'unbounded', minOccur => 0, type=>'Validator')
            ),
            new IMPL::DOM::Schema::Property(name => 'name')
        ),
        IMPL::DOM::Schema::ComplexType->new(type => 'SimpleType', nativeType => 'IMPL::DOM::Schema::SimpleType')->appendRange(
            IMPL::DOM::Schema::NodeList->new()->appendRange(
            	IMPL::DOM::Schema::Node->new(name => 'Property', type=>'Property', maxOccur=>'unbounded', minOccur=>0),
                IMPL::DOM::Schema::AnyNode->new(maxOccur => 'unbounded', minOccur => 0, type=>'Validator')
            ),
            new IMPL::DOM::Schema::Property(name => 'type'),
            new IMPL::DOM::Schema::Property(name => 'inflator', optional => 1, inflator => 'IMPL::DOM::Schema::InflateFactory')
        ),
        IMPL::DOM::Schema::ComplexType->new(type => 'SimpleNode', nativeType => 'IMPL::DOM::Schema::SimpleNode')->appendRange(
            IMPL::DOM::Schema::NodeList->new()->appendRange(
            	IMPL::DOM::Schema::Node->new(name => 'Property', type=>'Property', maxOccur=>'unbounded', minOccur=>0),
                IMPL::DOM::Schema::AnyNode->new(maxOccur => 'unbounded', minOccur => 0, type=>'Validator')
            ),
            new IMPL::DOM::Schema::Property(name => 'name'),
            new IMPL::DOM::Schema::Property(name => 'inflator', optional => 1, inflator => 'IMPL::DOM::Schema::InflateFactory')
        ),
        IMPL::DOM::Schema::ComplexType->new(type => 'Validator', nativeType => 'IMPL::DOM::Schema::Validator')->appendRange(
            IMPL::DOM::Schema::NodeList->new()->appendRange(
                IMPL::DOM::Schema::AnyNode->new(maxOccur => 'unbounded', minOccur => 0)
            )
        ),
        IMPL::DOM::Schema::ComplexType->new(type => 'Property', nativeType => 'IMPL::DOM::Schema::Property' )->appendRange(
        	IMPL::DOM::Schema::NodeList->new()->appendRange(
        		IMPL::DOM::Schema::AnyNode->new(maxOccur => 'unbounded', minOccur => 0)
        	),
        	IMPL::DOM::Schema::Property->new(name => 'name'),
        	new IMPL::DOM::Schema::Property(name => 'inflator', optional => 1, inflator => 'IMPL::DOM::Schema::InflateFactory')
        ),
        IMPL::DOM::Schema::SimpleType->new(type => 'Node', nativeType => 'IMPL::DOM::Schema::Node')->appendRange(
        	IMPL::DOM::Schema::Property->new(name => 'name'),
        	IMPL::DOM::Schema::Property->new(name => 'type')
        ),
        IMPL::DOM::Schema::SimpleType->new(type => 'AnyNode', nativeType => 'IMPL::DOM::Schema::AnyNode')
    );
    
    $schema->Process;
    
    return $schema;
}

1;

__END__

=pod

=head1 NAME

C<IMPL::DOM::Schema> -  .

=head1 DESCRIPTION

C<use base qw(IMPL::DOM::Document)>

DOM  -  ,    ,  
 .

=head1 METHODS

=over

=item C<< $obj->Process() >>

    .

=item C<< $obj->ResolveType($typeName) >>

   c  C<$typeName>.

=back

=head1 META SCHEMA

   ,       ,   

=begin code xml

<schema>
    <ComplexNode name="schema">
        <NodeSet>
            <Node minOcuur="0" maxOccur="unbounded" name="ComplexNode" type="ComplexNode"/>
            <Node minOcuur="0" maxOccur="unbounded" name="SimpleNode" type="SimpleNode"/>
            <Node minOcuur="0" maxOccur="unbounded" name="ComplexType" type="ComplexType"/>
            <Node minOcuur="0" maxOccur="unbounded" name="SimpleType" type="SimpleType"/>
            <SimpleNode minOcuur="0" maxOccur="unbounded" name="Node"/>
            <SimpleNode minOcuur="0" maxOccur="unbounded" name="Include"/>
        </NodeSet>
    </ComplexNode>
    
    <ComplexType type="NodeContainer">
        <NodeSet>
            <Node minOcuur="0" maxOccur="unbounded" name="ComplexNode" type="ComplexNode"/>
            <Node minOcuur="0" maxOccur="unbounded" name="SimpleNode" type="SimpleNode"/>
            <SimpleNode minOcuur="0" maxOccur="unbounded" name="Node"/>
        </NodeSet>
    </ComplexType>
    
    <ComplexType type="ComplexType">
        <NodeList>
            <Node name="NodeSet" type="NodeContainer" minOcuur=0/>
            <Node name="NodeList" type="NodeContainer" minOccur=0/>
            <AnyNode minOccur="0" maxOccur="unbounded"  type="Validator"/>
        </NodeList>
    </ComplexType>
    
    <ComplexType type="ComplexNode">
        <NodeList>
            <Node name="NodeSet" type="NodeContainer" minOcuur=0/>
            <Node name="NodeList" type="NodeContainer" minOccur=0/>
            <AnyNode minOccur="0" maxOccur="unbounded"  type="Validator"/>
        </NodeList>
    </ComplexType>
    
    <ComplexType type="SimpleNode">
        <NodeSet>
            <AnyNode minOccur=0 maxOccur="unbounded" type="Validator"/>
        </NodeSet>
    </ComplexType>
    
    <ComplexType type="SimpleType">
        <NodeSet>
            <AnyNode minOccur=0 maxOccur="unbounded" type="Validator"/>
        </NodeSet>
    </ComplexType>
    
    <ComplexType type="Validator">
        <NodeSet>
            <AnyNode minOccur=0 maxOccur="unbounded"/>
        </NodeSet>
    </ComplexType>
    
</schema>

=end code xml

=cut
