package Test::SQL::Schema;
use strict;
use warnings;

use parent qw(IMPL::Test::Unit);
__PACKAGE__->PassThroughArgs;

use IMPL::Class::Property;
use IMPL::Class::Property::Direct;

use IMPL::Test qw(test shared failed);

BEGIN {
    shared public property schemaDB => prop_all;
}

require IMPL::SQL::Schema;
require IMPL::SQL::Schema::Constraint::Unique;

use IMPL::SQL::Types qw(Integer Varchar);

test CreateSchema => sub {
    my ($this) = @_;
    
    my $schema = new IMPL::SQL::Schema(name => 'dbTest', version => 1) or failed "Failed to create schema";
    
    failed "Failed to set a schema name" unless $schema->name eq 'dbTest';
    failed "Failed to set a schema version" unless $schema->version == 1;
    
    $this->schemaDB($schema);
};

test AddTable => sub {
    my ($this) = @_;
    
    my $table = $this->schemaDB->AddTable({name => 'User'}) or failed "Failed to add a table to the schema";
    $table->InsertColumn({
        name => 'Id',
        type => Integer
    });
    $table->InsertColumn({
        name => 'Login',
        type => Varchar(255)
    });
    $table->InsertColumn({
        name => 'DisplayName',
        canBeNull => 1,
        type => Varchar(255)
    });
    $table->InsertColumn({
        name => 'RoleId',
        canBeNull => 1,
        type => Integer
    });
        
    my $colCount = @{$table->columns};
    
    failed "Failed to add columns", "Expected: 4", "Got: ".$colCount unless $colCount == 4;
    
    my $table2 = $this->schemaDB->AddTable({name => 'Role'});
    $table2->InsertColumn({
        name => 'Id',
        type => Integer
    });
    $table2->InsertColumn({
        name => 'Description',
        type => Varchar(255)
    });
    $table2->InsertColumn({
        name => 'ObsoleteId',
        type => Integer
    });
    
};

test SetPrimaryKey => sub {
	my ($this) = @_;
	
	my $tableUser = $this->schemaDB->GetTable('User');
	my $tableRole = $this->schemaDB->GetTable('Role');
	
	$tableUser->AddConstraint( pk => { columns => ['Id'], name => 'PK' });
	$tableRole->SetPrimaryKey('Id');
	
	$tableUser->primaryKey->HasColumn('Id') or failed "A primary key of 'User' table should contain 'Id' column";
	$tableRole->primaryKey->HasColumn('Id') or failed "A primary key of 'Role' table should contain 'Id' column";
	
};

test LinkTables => sub {
	my ($this) = @_;
	
	my $tableUser = $this->schemaDB->GetTable('User');
	my $tableRole = $this->schemaDB->GetTable('Role');
	
	$tableUser->LinkTo($tableRole,'RoleId');
	
	$tableUser->GetColumnConstraints('RoleId') == 1 or failed "Wrong constraints count for 'RoleId' column", $tableUser->GetColumnConstraints('RoleId');
};

test AddConstraint => sub {
    my ($this) = @_;
    
    my $table = $this->schemaDB->GetTable('Role') or failed "Failed to get a table";
    
    my $constraint = $table->AddConstraint(
        new IMPL::SQL::Schema::Constraint::Unique(
            name => 'Role_ObsoleteId_Uniq',
            table => $table,
            columns => ['ObsoleteId']
        )
    ) or failed "Failed to add constraint";
    
    failed "Failed to retrieve a constraint" unless ($table->GetColumnConstraints('ObsoleteId'))[0] == $constraint;
    
};

test RemoveConstraint => sub {
	my ($this) = @_;
	
	my $table = $this->schemaDB->GetTable('Role') or failed "Failed to get a table";
	my $constraint = $table->GetConstraint('Role_ObsoleteId_Uniq');
	
	eval {
		$table->RemoveColumn('ObsoleteId');
		1;
	} and failed "Should not remove column with constraint";
	
	$table->RemoveColumn('ObsoleteId','force');
    
    failed "A constraint remains alive after column deletion" unless $constraint->isDisposed;
	
};

test RemoveTable => sub {
	my ($this) = @_;
	
	my $table = $this->schemaDB->GetTable('Role') or failed "Failed to get a table";
	
	$this->schemaDB->RemoveTable('Role');
	
	$table->isDisposed or failed "A table remains alive after deletion";
	
	my $table2 = $this->schemaDB->GetTable('User');
	
	$table2->GetColumnConstraints('RoleId') == 0 or failed "A foreign key keept alive";
};

test Clone => sub {
	my ($this) = @_;
	
	my $clone1 = $this->schemaDB->Clone();
	
	$clone1->Dispose();
	
	$this->schemaDB->isDisposed and failed "An original schema should not be disposed";
};

test Dispose => sub {
    my ($this) = @_;
    
    $this->schemaDB->Dispose();
};


1;
