package Test::SQL::Schema;
use strict;
use warnings;

use base qw(IMPL::Test::Unit);
__PACKAGE__->PassThroughArgs;

use IMPL::Class::Property;
use IMPL::Class::Property::Direct;

use IMPL::Test qw(test shared failed);

BEGIN {
    shared public property schemaDB => prop_all;
}

require IMPL::SQL::Schema;
require IMPL::SQL::Schema::Constraint::Unique;

use IMPL::SQL::Types qw(Integer Varchar);

test CreateSchema => sub {
    my ($this) = @_;
    
    my $schema = new IMPL::SQL::Schema(Name => 'dbTest', Version => 1) or failed "Failed to create schema";
    
    failed "Failed to set a schema name" unless $schema->Name eq 'dbTest';
    failed "Failed to set a schema version" unless $schema->Version == 1;
    
    $this->schemaDB($schema);
};

test AddTable => sub {
    my ($this) = @_;
    
    my $table = $this->schemaDB->AddTable({Name => 'User'}) or failed "Failed to add a table to the schema";
    $table->InsertColumn({
        Name => 'Id',
        Type => Integer
    });
    $table->InsertColumn({
        Name => 'Login',
        Type => Varchar(255)
    });
    $table->InsertColumn({
        Name => 'DisplayName',
        CanBeNull => 1,
        Type => Varchar(255)
    });
    $table->InsertColumn({
        Name => 'RoleId',
        CanBeNull => 1,
        Type => Integer
    });
    
    $table->SetPrimaryKey('Id');
    
    my $colCount = @{$table->Columns};
    
    failed "Failed to add columns", "Expected: 4", "Got: ".$colCount unless $colCount == 4;
    failed "Failed to set a primary key" unless $table->PrimaryKey;
    
    my $table2 = $this->schemaDB->AddTable({Name => 'Role'});
    $table2->InsertColumn({
        Name => 'Id',
        Type => Integer
    });
    $table2->InsertColumn({
        Name => 'Description',
        Type => Varchar(255)
    });
    $table2->InsertColumn({
        Name => 'ObsoleteId',
        Type => Integer
    });
    
    $table2->SetPrimaryKey('Id');
    
    $table->LinkTo($table2,'RoleId');
};

test Constraints => sub {
    my ($this) = @_;
    
    my $table = $this->schemaDB->Tables->{Role} or failed "Failed to get a table";
    
    my $constraint = $table->AddConstraint(
        new IMPL::SQL::Schema::Constraint::Unique(
            Name => 'Role_ObsoleteId_Uniq',
            Table => $table,
            Columns => ['ObsoleteId']
        )
    ) or failed "Failed to add constraint";
    
    failed "Failed to retrieve a constraint" unless ($table->GetColumnConstraints('ObsoleteId'))[0] == $constraint;
    
    $table->RemoveColumn('ObsoleteId',1);
    
    failed "A constraint remains alive after column deletion" unless $constraint->isDisposed;
    
};

test Dispose => sub {
    my ($this) = @_;
    
    $this->schemaDB->Dispose();
};


1;
