package IMPL::DOM::Navigator::Builder;
use strict;
use warnings;

use base qw(IMPL::Object);
use IMPL::Class::Property;
use IMPL::Class::Property::Direct;
require IMPL::DOM::Navigator::SchemaNavigator;

BEGIN {
    private _direct property _schemaNavi => prop_all;
    private _direct property _nodesPath => prop_all;
    private _direct property _nodeCurrent => prop_all;
    private _direct property _docClass => prop_all;
    public _direct property Document => prop_get | owner_set;
}

sub CTOR {
    my ($this,$docClass,$schema) = @_;
    
    $this->{$_docClass} = $docClass;
    $this->{$_schemaNavi} = $schema ? IMPL::DOM::Navigator::SchemaNavigator->new($schema) : undef;
}

sub NavigateCreate {
    my ($this,$nodeName,%props) = @_;
    
    if (my $schemaNode = $this->{$_schemaNavi}->NavigateName($nodeName)) {
        my $class = $schemaNode->can('nativeType') ? $schemaNode->nativeType : 'IMPL::DOM::Node';
        $this->inflateProperties($schemaNode,\%props);
        
        my $node;
        if (! $this->{$Document}) {
            $node = $this->{$Document} = $this->{$_docClass}->new(nodeName => $nodeName,%props);
        } else {
            die new IMPL::InvalidOperationException('Can\t create a second top level element') unless $this->{$_nodeCurrent};
            $node = $this->{$Document}->Create($nodeName,$class,\%props);
            push @{$this->{$_nodesPath}}, $this->{$_nodeCurrent};
            $this->{$_nodeCurrent}->appendChild($node);
        }
        
        $this->{$_nodeCurrent} = $node;
        
        return $node;
    } else {
        die new IMPL::InvalidOperationException("The specified node is undefined", $nodeName);
    }
}

sub inflateProperties {
	my ($this,$schemaNode,$refProps) = @_;
	
	$refProps->{$_->name} = $_->inflator->new($refProps->{$_->name})
		foreach $schemaNode->selectNodes(
			sub {
				$_->nodeName eq 'Property' and exists $refProps->{$_->name} and $_->inflator 
			}
		);
}

sub inflateValue {
	my ($this,$value) = @_;
	my $schemaNode = $this->{$_schemaNavi}->Current;
	if ($schemaNode->can('inflator') and my $inflator = $schemaNode->inflator) {
		return $inflator->new($value);
	} else {
		return $value;
	}
}

sub Back {
    my ($this) = @_;
    
    $this->{$_schemaNavi}->SchemaBack();
    $this->{$_nodeCurrent} = pop @{$this->{$_nodesPath}};
}

1;

__END__

=pod

=head1 NAME

C< IMPL::DOM::Navigator::Builder > - ,     .

=head1 SYNOPSIS

=begin code

my $builder = new IMPL::DOM::Navigator::Builder(new MyApp::Document,$schema);
my $reader = new IMPL::DOM::XMLReader(Navigator => $builder);

$reader->ParseFile("document.xml");

my @errors = $schema->Validate($builder->Document);

=end code

=head1 DESCRIPTION

 DOM    .    
      C<IMPL::DOM::XMLReader>.

=head1 METHODS

=over

=item C< CTOR($classDocument,$schema) >

  ,      ( , 
L<IMPL::Object::Factory>)  .     
   .

=item C< NavigateCreate($nodeName,\%props) >

         .     
   ,   .

       ,     
       .  
     C<< IMPL::DOM::Document->Create >>

       C<props>,   
      C<nodeName>,   .

=item C< Document >

,       .

=back

=cut
