package Test::Web::ViewSelector;
use strict;

use Scalar::Util qw(reftype);
use IMPL::Test qw(test assert assertarray);
use IMPL::declare {
    require => {
        TTView => 'IMPL::Web::Handler::TTView'        
    },
    base => [
        'IMPL::Test::Unit' => '@_'
    ]
};

test TestParseRules => sub {
    my $rule = TTView->ParseRule('/category/item => item.html');
    
    assert(reftype($rule) eq 'HASH');
    assert(reftype($rule->{selector}) eq 'ARRAY');
    assertarray([map $_->{name}, @{$rule->{selector}}], [qw(category item)]);
    assert($rule->{data} eq 'item.html');
    assert($rule->{weight} == 2);
    
    $rule = TTView->ParseRule('category//item => item.html');
    
    assert(reftype($rule) eq 'HASH');
    assert(reftype($rule->{selector}) eq 'ARRAY');
    assertarray([map $_->{name}, @{$rule->{selector}}], [undef, 'category', undef ,'item']);
    assert($rule->{data} eq 'item.html');
    assert($rule->{weight} == 2);
    
    $rule = TTView->ParseRule('///category//item///// => item.html');
    # trailing slashes must be ignored
    assertarray([map $_->{name}, @{$rule->{selector}}], [undef,undef, 'category', undef ,'item']);
    
    $rule = TTView->ParseRule('{cat:\w+}@OrgClass/products/@My::PoductClass/view => view.html');
    
    assert(reftype($rule) eq 'HASH');
    assert(reftype($rule->{selector}) eq 'ARRAY');
    assert($rule->{data} eq 'view.html');
    assert($rule->{weight} == 4);
    assert($rule->{selector}[0]->{any});
    assert($rule->{selector}[1]->{rx});
    assert($rule->{selector}[1]->{var} eq 'cat');
    assert($rule->{selector}[1]->{class} eq 'OrgClass');
    assert($rule->{selector}[3]->{class} eq 'My::PoductClass');
    assertarray([map $_->{name}, @{$rule->{selector}}], [undef,undef, 'products', undef ,'view']);
    
    $rule = TTView->ParseRule('/ => index.html');
    assert($rule->{weight} == 0);
};

test TestNamesMatch => sub {
    my @rules = map TTView->ParseRule($_),
        'view => view.html', # weight 1
        'shoes/view => shoes/view.html', # weight 2
        '/root/org/items/add => add.html'; # weight 4
        
    assert(
        TTView->MatchPath(
            [ map { name => $_ }, qw(root view)],
            \@rules
        ) eq 'view.html'
    );
    
    assert(
        TTView->MatchPath(
            [ map { name => $_ }, qw(root shoes view)],
            \@rules
        ) eq 'shoes/view.html'
    );
    
    assert(
        TTView->MatchPath(
            [ map { name => $_ }, qw(root org products shoes view)],
            \@rules
        ) eq 'shoes/view.html'
    );
    
    assert(
        TTView->MatchPath(
            [ map { name => $_ }, qw(root org items add)],
            \@rules
        ) eq 'add.html'
    );
};

{
    package Test::Web::ViewSelector::Container;
    
    package Test::Web::ViewSelector::Orgs;
    use IMPL::declare {
        base => ['-Test::Web::ViewSelector::Container' => undef]
    };
}

test TestComplexMatch => sub {
    my @rules = map TTView->ParseRule($_),
        '{container:.*}@Test::Web::ViewSelector::Container/{item:.*}/{action:.*} => {container}/{item}/{action}.html', # weight 3
        '/root//orgs/{org:.*}/info => orgs/{org}.html', # weight 4
        '@Test::Web::ViewSelector::Container => container.html'; 

    my $path = [
        { name => 'root'},
        { name => 'list', class => 'Test::Web::ViewSelector::Container'},
        { name => 'hp' },
        { name => 'info'}
    ];
    
    my $result = TTView->MatchPath($path,\@rules); 
    my $expected = 'list/hp/info.html';

    assert( $result eq $expected, "Expected: $expected", "Got: $result" );
    
    $path = [
        { name => 'root'},
        { name => 'orgs', class => 'Test::Web::ViewSelector::Orgs'},
        { name => 'ms' },
        { name => 'info'}
    ];
    $result = TTView->MatchPath($path,\@rules); 
    $expected = 'orgs/ms.html';

    assert( $result eq $expected, "Expected: $expected", "Got: $result" );
    
    $path = [
        { name => 'root'},
        { name => 'service'},
        { name => 'orgs', class => 'Test::Web::ViewSelector::Container' }
    ];
    $result = TTView->MatchPath($path,\@rules); 
    $expected = 'container.html';

    assert( $result eq $expected, "Expected: $expected", "Got: $result" );
};

1;