package IMPL::Security::Context;
use strict;
use warnings;

use parent qw(IMPL::Object IMPL::Object::Autofill);

__PACKAGE__->PassThroughArgs;

use IMPL::Class::Property;

require IMPL::Security::Principal;

my $current;
my $nobody;

BEGIN {
    public property principal => prop_get;
    public property rolesAssigned => prop_all | prop_list;
    public property auth => prop_all;
    public property authority => prop_all;
}

sub CTOR {
    my ($this) = @_;
    
    die new IMPL::InvalidArgumentException("The parameter is required", 'principal') unless $this->principal;
}

sub Impersonate {
    my ($this,$code) = @_;
    
    my $old = $current;
    $current = $this;
    my $result;
    my $e;
    
    {
        local $@;
        eval {
            $result = $code->();
        };
        $e = $@;
    }
    $current = $old;
    if($e) {
        die $e;
    } else {
        return $result;
    }
}

sub Apply {
    my ($this) = @_;
    
    $current = $this;
}

sub isTrusted {
    my ($this) = @_;
    
    if (my $auth = $this->auth) {
        return $auth->isTrusted;
    } else {
        return 0;
    }
}

sub nobody {
    my ($self) = @_;
    $nobody = $self->new(principal => IMPL::Security::Principal->nobody) unless $nobody;
    $nobody;
}

sub current {
    my ($self) = @_;
    
    $current = __PACKAGE__->nobody unless $current;
    $current;
}

sub Satisfy {
    my ($this,@roles) = @_;
    
    my $roleEffective = new IMPL::Security::Role ( _effective => scalar $this->rolesAssigned );
    
    return $roleEffective->Satisfy(@roles);
}

1;

__END__

=pod

=head1 NAME

C<IMPL::Security::Context> - контекст безопасности.

=head1 SINOPSYS

=begin code

my $context = IMPL::Security::Context->nobody;

my $result = $context->Impersonate(
    sub {
        # do some untrusted code
    }
);

=end code

=head1 DESCRIPTION

C<[Autofill]>

Являет собой контекст безопасности, описывает пользователя и привелегии, так же
у программы есть текущий контекст безопасности, по умолчанию он C<nobody>.

=head1 MEMBERS

=over

=item C<CTOR(%props)>

Создает объект и заполняет его свойствами.

=item C<[get] principal>

Идентификатор пользователя, владельца контекста.

=item C<[get] rolesAssigned>

Список назначенных (активных) ролей пользователю.

=item C<[get] auth>

Объект асторизации C<IMPL::Security::Auth>, использованный при создании текущего контекста.

=item C<[static,get] authority>

Источник данных безопасности, породивший данный контекст.

=item C<[get] isTrusted>

Возвращает значение является ли контекст доверенным, тоесть сессия аутетифицирована.

=item C<Impersonate($code)>

Делает контекст текущим и выполняет в нем функцию по ссылке C<$code>. По окончании
выполнения, контекст восстанавливается.

=item C<Apply()>

Заменяет текущий контекст на себя, но до конца действия метода C<Impersonate>, если
таковой был вызван.

=item C<[static,get] nobody>

Контекст для неаутентифицированных пользователей, минимум прав.

=item C<[static,get] current>

Текущий контекст.

=back

=cut
