package IMPL::Web::AutoLocator;
use strict;

use IMPL::lang qw(:declare :constants :hash);
use URI;
use URI::Escape;
use IMPL::declare {
	require => {
		Exception => 'IMPL::Exception',
		ArgumentException => '-IMPL::InvalidArgumentException'
	},
	base => [
        'IMPL::Object' => undef,
        'IMPL::Object::Autofill' => '@_',
        'IMPL::Object::Serializable' => '@_'
	]
};

BEGIN {
	public property base => PROP_GET | PROP_OWNERSET;
	public property view => PROP_ALL;
	public property query => PROP_ALL;
	public property hash => PROP_ALL;
}

sub Fetch {
	my $this = shift;
	my $child = shift or die ArgumentException->new("a child resource identifier is required");
	die ArgumentException->new("a child resource can't be a reference") if ref $child;
	
	# safe
	$child = uri_escape($child);
	
	my %args;
	
	$args{base} = $this->base =~ /\/$/ ? $this->base . $child : $this->base . '/' . $child;
	$args{view} = $this->view if $this->view;
	$args{hash} = $this->hash if $this->hash;
	
	if (@_) {
		my $query = shift;
		
		$args{query} = ref $query eq 'HASH' ? hashMerge($this->query,$query) : $query;
	}
	
	return $this->new(%args);
}

sub SetView {
	my ($this,$newView) = @_;
	
	$this->view($newView);
	
	return $this;
}

sub url {
	my ($this) = @_;
	
	my $url = URI->new($this->view ? $this->base . "." . $this->view : $this->base);
	$url->query_form($this->query);
	$url->fragment($this->hash);
	
	return $url;
}

sub AUTOLOAD {
	our $AUTOLOAD;
	
	(my $method) = ($AUTOLOAD =~ m/(\w+)$/);
	
	return if $method eq 'DESTROY';
	
	my $this = shift;
	return $this->Fetch($method,@_);
}



1;

__END__

=head1 NAME

=head1 SYNOPSIS

C<IMPL::Web::AutoLocator> - Обертка вокруг адреса ресурса.

=begin code

use IMPL::require {
	Locator => 'IMPL::Web::Locator'
};

my $bugTracker = Locator->new(base => "http://myhost.org/bugzilla")->SetView("cgi");

my $bug = $bugTracker->show_bug({id = 1});

my $wikiPages = Locator->new(base => "http://myhost.org/wiki/bin/view");

my $page = $wiki->Main->HowTo;

=end code

=head1 DESCRIPTION

Для удобстав навигации по ресурсам, полностью отражает классическую структуру иерархически
организованных ресурсов. позволяет гибко работать с параметрами запроса и хешем. Для постоты
чтения реализует метод C<AUTOLOAD> для доступа к дочерним ресурсам.

=head1 MEMBERS

=head2 C<CTOR(base => $url,view => $extension, query => $hashQuery, hash => $fragment)>

Создает новый объект расположение. Позволяет задать путь, расширение, параметры запроса и фрагмент ресурса.

=head2 C<Fetch($child[,$query])>

Получает расположение дочернего ресурса. При этом моздается новый объект адреса ресурса.

=head2 C<AUTLOAD>

Перенаправляет вызовы методов в метод C<Fetch> передавая первым параметром имя метода.

=cut

