package IMPL::Web::Security;
use strict;
use parent qw(IMPL::Object IMPL::Security IMPL::Object::Autofill);

require IMPL::Web::Security::Session;

use IMPL::Class::Property;
use IMPL::Security::Auth qw(:Const);

__PACKAGE__->PassThroughArgs;

BEGIN {
    public property sourceUser => prop_all;
    public property sourceSession => prop_all;
}

sub CTOR {
    my ($this) = @_;
    
    die new IMPL::InvalidArgumentException("An argument is required",'sourceUser') unless $this->sourceUser;
    die new IMPL::InvalidArgumentException("An argument is required",'sourceSession') unless $this->sourceSession;
}

sub AuthUser {
    my ($this,$name,$package,$challenge) = @_;
    
    my $user = $this->sourceUser->find({name => $name}) or return { status => AUTH_FAIL, answer => "Can't find a user '$name'" };
    
    my $auth;    
    if ( my $secData = $user->secData($package) ) {
        $auth = $package->new($secData);
    } else {
        die new IMPL::SecurityException("Authentication failed","A sec data for the $package isn't found");
    }
    
    my ($status,$answer) = $auth->DoAuth($challenge);
    
    return {
        status => $status,
        answer => $answer,
        context => $this->MakeContext( $user, [$user->roles], $auth )
    }
}

sub MakeContext {
    my ($this,$principal,$roles,$auth) = @_;
    
    return $this->sourceSession->create(
        {
            principal => $principal,
            rolesAssigned => $roles,
            auth => $auth
        }
    );
}

1;

__END__

=pod

=head1 NAME

C<IMPL::Web::Security> Модуль для аутентификации и авторизации веб запроса.

=head1 SINOPSYS

=begin code xml

<security type='IMPL::Config::Activator'>
    <factory>IMPL::Web::Security</factory>
    <parameters type='HASH'>
        <sessionFactory type='IMPL::Object::Factory'>
            <factory type='IMPL::Object::Factory'>App::Data::Session</factory>
            <method>insert</method>
        </sessionFactory>
    </parameters>
</security>

=end code xml

=head1 DESCRIPTION

Отвечает за инфраструктуру аутентификации и авторизации запросов. Основная особенность
заключается в том, что запросы приходят через значительные интевалы времени, хотя и
относятся к одной логической транзакции. В промежутках между запросами сервер не
сохраняет свое состояние. Поэтому при каждом обращении сервер восстанавливает
контекст безопасности.

C<IMPL::Web::Session> Объект обеспечивающий сохранение состояния в рамках одной сессии
пользователя. Кроме контекста безопасности хранит дополнительние данные, которые необходимо
сохранить между обработкой запросов.

C<IMPL::Web::User> Объект, устанавливающий связь между идентификатором пользователя
C<IMPL::Security::Principal>, его ролями и данными безопасности для создания объектов
аутентификации C<IMPL::Security::Auth>.

=head1 MEMBERS

=cut
