package IMPL::Code::Binding;
use strict;

use IMPL::require {
	Exception => 'IMPL::Exception',
	ArgumentException => '-IMPL::ArgumentException'
};

sub new {
	my ($self,$expr,$vars) = @_;
	
	$vars ||= [];
	
	die ArgumentException->new( vars => 'A reference to an array is required')
	   unless ref $vars eq 'ARRAY';
	
	m/^\w+$/ or die ArgumentException->new( vars => 'A valid variable name is required', $_ )
	   foreach @$vars;    
	
	my $varnames = join (',', map { "\$$_" } @$vars);   
    
    my $code = <<CODE;
    sub {
        my ($varnames) = \@_;
        $text
    }
CODE
    my $method = eval $code; #$compiler_env->reval($code,'strict');
    
    return $method;
}

1;

__END__

=pod

=head1 NAME

C<IMPL::Code::Binding> - превращает выражения в процедуру.

=head1 SYNOPSIS

=begin code

use IMPL::require {
	Binding => 'IMPL::Code::Binding'
}

my $person = DB->SearchPerson({name => 'Peter'})->First;

my $bind = Binding->new(
    [qw(obj)] =>
    q{ $obj->addresses->[0]->country->code }
);

print $bind->($person);

=end

=head1 DESCRIPTION

Позвоялет преобразовать выражение в функцию, которую можно будет многократно
использовать для получения значения выражения.

Выражение параметризуется произвольным количеством именованных параметров,
которые будут доступны внутри выражения как переменные. При создании связывателя
в конструктор передается выражение связывания, ссылка насписок из параметров.

При создания связывателя будет проверен синтаксис, и если в выражении допущена
ошибка, возникнет исключение.

Данный класс не является безопасным при создании связывателей из ненадежных
источников, поскольку внутри будет выполнен C<eval>.

=head1 MEMBERS

=head2 C<new(\@vars,$expression)>

Возвращает ссылку на процедуру.

=cut