package IMPL::AppException;
use strict;
use mro 'c3';
use overload
    '""' => 'ToString',
    'bool'     =>   sub { return 1; },
    'fallback' => 1;

use Carp qw(longmess shortmess);
use Scalar::Util qw(refaddr);

use IMPL::Const qw(:prop);
use IMPL::Resources::Strings {
	message => "Application exception"
};

use IMPL::declare {
    base => [
        'IMPL::Object' => undef
    ],
    props => [
        source => PROP_RO,
        callStack => PROP_RO,
    ]
};

sub new {
	my $self = shift;
	
	my $instance = $self->next::method(@_);
	
	$instance->source(shortmess);
	$instance->callStack(longmess);
	
    return $instance;
}

sub ToString {
	my ($this) = @_;
	
	return join("\n", $this->message, $this->callStack);
}

sub throw {
	my $self = shift;
	
	die ref $self ? $self : $self->new(@_);
}

1;

__END__

=pod

=head1 NAME

C<IMPL::AppException> - исключение приложения.

=head1 SYNOPSIS

=begin code

package MyException;
use strict;

use IMPL::Const qw(:prop);
use IMPL::declare {
    base => [
        'IMPL::AppException' => undef
    ],
    props => [
        level => PROP_RO
    ] 
};

use IMPL::Resources::Strings {
    message => "Something wrong at level %level%"
};

sub CTOR {
    my ($this,$level) = @_;
    
    $this->level($level);
}

=end code

=head1 DESCRIPTION

Для описания собственных исключений в качестве базового класса должен
использоваться C<IMPL::AppException> поскольку он позволяет использовать
C<IMPL::declare> и объявлять свойства.

C<IMPL::Exception> также является классом для исключений, однако поскольку
он используется в базовых механизмах библиотеки, то в нем не реализованы
механизмы для описания свойсвт.

Исключение имеет свойство C<message> которое возвращает текст с описанием
проблемы, данное свойство можно реализовать с использованием
C<IMPL::Resources::Strings> для реализации поддержки нескольких языков.

Особенностью тсключений также является то, что при их создании автоматически
фиксируется место, где оно было создано и свойства C<source> и C<callStack>
заполняются автоматически.

Для исключений переопределены операторы приведения к строке и к булевому
значению.

=head1 MEMBERS

=head2 C<[op]new(@args)>

Оператор создающий новый экземпляр исключения, сначала создает экземпляр
исключения, затем заполняет свойства C<source>, C<callStack>.

=head2 C<[op]throw(@args)>

Создает объект исключения и бросает его.

=begin code

throw MyException(10);
MyException->throw(10); # ditto

=end code

=head2 C<[get]source>

Строка с описанием в каком файле и где произошло исключение. см. C<Carp>

=head2 C<[get]callStack>

Строка со стеком вызовов в момент возникновения исключения. см. C<Carp>

=head2 C<[get]message>

Возвращает описание исключения.

=head2 C<ToString()>

Возвращает текстовое представление, как правило это C<message> и C<callStack>.

=cut