﻿using Implab.Diagnostics;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Web;

namespace Implab {
    /// <summary>
    /// Объект, поддерживающий освобождение ресурсов.
    /// </summary>
    public class Disposable : IDisposable {
        
        bool m_disposed;

        public event EventHandler Disposed;

        public bool IsDisposed {
            get { return m_disposed; }
        }

        protected void AssertNotDisposed() {
            if (m_disposed)
                throw new ObjectDisposedException(this.ToString());
        }
        /// <summary>
        /// Переводит объект в состояние <c>Disposed</c> и вызывает событие <see cref="Disposed"/>
        /// </summary>
        /// <param name="disposing">Признак того, что нужно освободить ресурсы, иначе данный метод
        /// вызван сборщиком мусора и нужно освобождать ТОЛЬКО неуправляемые ресурсы ТОЛЬКО этого
        /// объекта.</param>
        /// <remarks>
        /// Данный метод осуществляет проверку того, что объект уже был освобожден, чтобы не вызывать
        /// событие <see cref="Disposed"/>. Не поддерживает многопоточность.
        /// </remarks>
        protected virtual void Dispose(bool disposing) {
            if (disposing && !m_disposed) {
                m_disposed = true;
                
                EventHandler temp = Disposed;
                if (temp != null) 
                    temp(this,EventArgs.Empty);
            }
        }
        public void Dispose() {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// Записывает сообщение об утечке объекта.
        /// </summary>
        protected virtual void ReportObjectLeaks() {
            TraceLog.TraceWarning("The object is marked as disposable but isn't disposed properly: {0}", this);
        }

        ~Disposable() {
            Dispose(false);
            ReportObjectLeaks();
        }
    }
}