using System;
using System.Collections.Generic;
using System.Reflection;
using System.Diagnostics;
using System.Threading;

namespace Implab {

    public delegate void ErrorHandler(Exception e);
    public delegate T ErrorHandler<out T>(Exception e);
    public delegate void ResultHandler<in T>(T result);
    public delegate TNew ResultMapper<in TSrc, out TNew>(TSrc result);
    public delegate Promise<TNew> ChainedOperation<in TSrc, TNew>(TSrc result);

    /// <summary>
    /// Класс для асинхронного получения результатов. Так называемое "обещание".
    /// </summary>
    /// <typeparam name="T">Тип получаемого результата</typeparam>
    /// <remarks>
    /// <para>Сервис при обращении к его методу дает обещаиние о выполнении операции,
    /// клиент получив такое обещание может установить ряд обратных вызово для получения
    /// событий выполнения обещания, тоесть завершения операции и предоставлении результатов.</para>
    /// <para>
    /// Обещение может быть как выполнено, так и выполнено с ошибкой. Для подписки на
    /// данные события клиент должен использовать методы <c>Then</c>.
    /// </para>
    /// <para>
    /// Сервис, в свою очередь, по окончанию выполнения операции (возможно с ошибкой),
    /// использует методы <c>Resolve</c> либо <c>Reject</c> для оповещения клиетна о
    /// выполнении обещания.
    /// </para>
    /// <para>
    /// Если сервер успел выполнить обещание еще до того, как клиент на него подписался,
    /// то в момент подписки клиента будут вызваны соответсвующие события в синхронном
    /// режиме и клиент будет оповещен в любом случае. Иначе, обработчики добавляются в
    /// список в порядке подписания и в этом же порядке они будут вызваны при выполнении
    /// обещания.
    /// </para>
    /// <para>
    /// Обрабатывая результаты обещания можно преобразовывать результаты либо инициировать
    /// связанные асинхронные операции, которые также возвращают обещания. Для этого следует
    /// использовать соответствующую форму методе <c>Then</c>.
    /// </para>
    /// <para>
    /// Также хорошим правилом является то, что <c>Resolve</c> и <c>Reject</c> должен вызывать
    /// только инициатор обещания иначе могут возникнуть противоречия.
    /// </para>
    /// </remarks>
    public class Promise<T> : IPromise {

        struct ResultHandlerInfo {
            public ResultHandler<T> resultHandler;
            public ErrorHandler errorHandler;
        }

        readonly IPromise m_parent;

        LinkedList<ResultHandlerInfo> m_resultHandlers = new LinkedList<ResultHandlerInfo>();
        LinkedList<Action> m_cancelHandlers = new LinkedList<Action>();

        readonly object m_lock = new Object();
        readonly bool m_cancellable;
        int m_childrenCount = 0;

        PromiseState m_state;
        T m_result;
        Exception m_error;

        public Promise() {
            m_cancellable = true;
        }

        public Promise(IPromise parent, bool cancellable) {
            m_cancellable = cancellable;
            m_parent = parent;
            if (parent != null)
                parent.HandleCancelled(InternalCancel);
        }

        void InternalCancel() {
            // don't try to cancel parent :)
            Cancel(false);
        }

        /// <summary>
        /// Выполняет обещание, сообщая об успешном выполнении.
        /// </summary>
        /// <param name="result">Результат выполнения.</param>
        /// <exception cref="InvalidOperationException">Данное обещание уже выполнено</exception>
        public void Resolve(T result) {
            lock (m_lock) {
                if (m_state == PromiseState.Cancelled)
                    return;
                if (m_state != PromiseState.Unresolved)
                    throw new InvalidOperationException("The promise is already resolved");
                m_result = result;
                m_state = PromiseState.Resolved;
            }

            OnStateChanged();
        }

        /// <summary>
        /// Выполняет обещание, сообщая об ошибке
        /// </summary>
        /// <param name="error">Исключение возникшее при выполнении операции</param>
        /// <exception cref="InvalidOperationException">Данное обещание уже выполнено</exception>
        public void Reject(Exception error) {
            lock (m_lock) {
                if (m_state == PromiseState.Cancelled)
                    return;
                if (m_state != PromiseState.Unresolved)
                    throw new InvalidOperationException("The promise is already resolved");
                m_error = error;
                m_state = PromiseState.Rejected;
            }

            OnStateChanged();
        }

        /// <summary>
        /// Отменяет операцию, если это возможно.
        /// </summary>
        /// <returns><c>true</c> Операция была отменена, обработчики не будут вызваны.<c>false</c> отмена не возможна, поскольку обещание уже выполнено и обработчики отработали.</returns>
        public bool Cancel() {
            return Cancel(true);
        }

        /// <summary>
        /// Adds new handlers to this promise.
        /// </summary>
        /// <param name="success">The handler of the successfully completed operation.
        /// This handler will recieve an operation result as a parameter.</param>
        /// <param name="error">Handles an exception that may occur during the operation.</param>
        /// <returns>The new promise chained to this one.</returns>
        public Promise<T> Then(ResultHandler<T> success, ErrorHandler error) {
            if (success == null && error == null)
                return this;

            var medium = new Promise<T>(this, true);

            var handlerInfo = new ResultHandlerInfo();

            if (success != null)
                handlerInfo.resultHandler = x => {
                    success(x);
                    medium.Resolve(x);
                };
            else
                handlerInfo.resultHandler = medium.Resolve;

            if (error != null)
                handlerInfo.errorHandler = x => {
                    try {
                        error(x);
                    } catch { }
                    medium.Reject(x);
                };
            else
                handlerInfo.errorHandler = medium.Reject;

            AddHandler(handlerInfo);

            return medium;
        }

        /// <summary>
        /// Adds new handlers to this promise.
        /// </summary>
        /// <param name="success">The handler of the successfully completed operation.
        /// This handler will recieve an operation result as a parameter.</param>
        /// <param name="error">Handles an exception that may occur during the operation and returns the value which will be used as the result of the operation.</param>
        /// <returns>The new promise chained to this one.</returns>
        public Promise<T> Then(ResultHandler<T> success, ErrorHandler<T> error) {
            if (success == null && error == null)
                return this;

            var medium = new Promise<T>(this, true);

            var handlerInfo = new ResultHandlerInfo();

            if (success != null)
                handlerInfo.resultHandler = x => {
                    success(x);
                    medium.Resolve(x);
                };
            else
                handlerInfo.resultHandler = medium.Resolve;

            if (error != null)
                handlerInfo.errorHandler = x => {
                    try {
                        medium.Resolve(error(x));
                    } catch { }
                    medium.Reject(x);
                };
            else
                handlerInfo.errorHandler = medium.Reject;

            AddHandler(handlerInfo);

            return medium;
        }


        public Promise<T> Then(ResultHandler<T> success) {
            if (success == null)
                return this;

            var medium = new Promise<T>(this, true);

            var handlerInfo = new ResultHandlerInfo();

            if (success != null)
                handlerInfo.resultHandler = x => {
                    success(x);
                    medium.Resolve(x);
                };
            else
                handlerInfo.resultHandler = medium.Resolve;

            handlerInfo.errorHandler = medium.Reject;

            AddHandler(handlerInfo);

            return medium;
        }

        public Promise<T> Error(ErrorHandler error) {
            return Then(null, error);
        }

        /// <summary>
        /// Handles error and allows to keep the promise.
        /// </summary>
        /// <remarks>
        /// If the specified handler throws an exception, this exception will be used to reject the promise.
        /// </remarks>
        /// <param name="handler">The error handler which returns the result of the promise.</param>
        /// <returns>New promise.</returns>
        public Promise<T> Error(ErrorHandler<T> handler) {
            if (handler == null)
                return this;

            var medium = new Promise<T>(this, true);

            AddHandler(new ResultHandlerInfo {
                errorHandler = e => {
                    try {
                        medium.Resolve(handler(e));
                    } catch (Exception e2) {
                        medium.Reject(e2);
                    }
                }
            });

            return medium;
        }

        public Promise<T> Anyway(Action handler) {
            if (handler == null)
                return this;

            var medium = new Promise<T>();

            AddHandler(new ResultHandlerInfo {
                resultHandler = x => {
                    // to avoid handler being called multiple times we handle exception by ourselfs
                    try {
                        handler();
                        medium.Resolve(x);
                    } catch (Exception e) {
                        medium.Reject(e);
                    }
                },
                errorHandler = x => {
                    try {
                        handler();
                    } catch { }
                    medium.Reject(x);
                }
            });

            return medium;
        }

        /// <summary>
        /// Позволяет преобразовать результат выполения операции к новому типу.
        /// </summary>
        /// <typeparam name="TNew">Новый тип результата.</typeparam>
        /// <param name="mapper">Преобразование результата к новому типу.</param>
        /// <param name="error">Обработчик ошибки. Данный обработчик получит
        /// исключение возникшее при выполнении операции.</param>
        /// <returns>Новое обещание, которое будет выполнено при выполнении исходного обещания.</returns>
        public Promise<TNew> Map<TNew>(ResultMapper<T, TNew> mapper, ErrorHandler error) {
            if (mapper == null)
                throw new ArgumentNullException("mapper");

            // создаем прицепленное обещание
            var chained = new Promise<TNew>();

            AddHandler(new ResultHandlerInfo() {
                resultHandler = result => chained.Resolve(mapper(result)),
                errorHandler = delegate(Exception e) {
                    if (error != null)
                        try {
                            error(e);
                        } catch { }
                    // в случае ошибки нужно передать исключение дальше по цепочке
                    chained.Reject(e);
                }
            });

            return chained;
        }

        public Promise<TNew> Map<TNew>(ResultMapper<T, TNew> mapper) {
            return Map(mapper, null);
        }

        /// <summary>
        /// Сцепляет несколько аснхронных операций. Указанная асинхронная операция будет вызвана после
        /// выполнения текущей, а результат текущей операции может быть использован для инициализации
        /// новой операции.
        /// </summary>
        /// <typeparam name="TNew">Тип результата указанной асинхронной операции.</typeparam>
        /// <param name="chained">Асинхронная операция, которая должна будет начаться после выполнения текущей.</param>
        /// <param name="error">Обработчик ошибки. Данный обработчик получит
        /// исключение возникшее при выполнении текуещй операции.</param>
        /// <returns>Новое обещание, которое будет выполнено по окончанию указанной аснхронной операции.</returns>
        public Promise<TNew> Chain<TNew>(ChainedOperation<T, TNew> chained, ErrorHandler error) {

            // проблема в том, что на момент связывания еще не начата асинхронная операция, поэтому нужно
            // создать посредника, к которому будут подвызяваться следующие обработчики.
            // когда будет выполнена реальная асинхронная операция, она обратиться к посреднику, чтобы
            // передать через него результаты работы.
            var medium = new Promise<TNew>(this, true);

            AddHandler(new ResultHandlerInfo {
                resultHandler = delegate(T result) {
                    if (medium.State == PromiseState.Cancelled)
                        return;

                    var promise = chained(result);

                    // notify chained operation that it's not needed
                    medium.Cancelled(() => promise.Cancel());
                    promise.Then(
                        x => medium.Resolve(x),
                        e => medium.Reject(e)
                    );
                },
                errorHandler = delegate(Exception e) {
                    if (error != null)
                        error(e);
                    // в случае ошибки нужно передать исключение дальше по цепочке
                    medium.Reject(e);
                }
            });

            return medium;
        }

        public Promise<TNew> Chain<TNew>(ChainedOperation<T, TNew> chained) {
            return Chain(chained, null);
        }

        public Promise<T> Cancelled(Action handler) {
            if (handler == null)
                return this;
            lock (m_lock) {
                if (m_state == PromiseState.Unresolved)
                    m_cancelHandlers.AddLast(handler);
                else if (m_state == PromiseState.Cancelled)
                    handler();
            }
            return this;
        }

        public void HandleCancelled(Action handler) {
            Cancelled(handler);
        }

        /// <summary>
        /// Дожидается отложенного обещания и в случае успеха, возвращает
        /// его, результат, в противном случае бросает исключение.
        /// </summary>
        /// <remarks>
        /// <para>
        /// Если ожидание обещания было прервано по таймауту, это не значит,
        /// что обещание было отменено или что-то в этом роде, это только
        /// означает, что мы его не дождались, однако все зарегистрированные
        /// обработчики, как были так остались и они будут вызваны, когда
        /// обещание будет выполнено.
        /// </para>
        /// <para>
        /// Такое поведение вполне оправдано поскольку таймаут может истечь
        /// в тот момент, когда началась обработка цепочки обработчиков, и
        /// к тому же текущее обещание может стоять в цепочке обещаний и его
        /// отклонение может привести к непрогнозируемому результату.
        /// </para>
        /// </remarks>
        /// <param name="timeout">Время ожидания</param>
        /// <returns>Результат выполнения обещания</returns>
        public T Join(int timeout) {
            var evt = new ManualResetEvent(false);
            Anyway(() => evt.Set());
            Cancelled(() => evt.Set());

            if (!evt.WaitOne(timeout, true))
                throw new TimeoutException();

            switch (State) {
                case PromiseState.Resolved:
                    return m_result;
                case PromiseState.Cancelled:
                    throw new OperationCanceledException();
                case PromiseState.Rejected:
                    throw new TargetInvocationException(m_error);
                default:
                    throw new ApplicationException(String.Format("Invalid promise state {0}", State));
            }
        }

        public T Join() {
            return Join(Timeout.Infinite);
        }

        void AddHandler(ResultHandlerInfo handler) {
            bool invokeRequired = false;

            lock (m_lock) {
                m_childrenCount++;
                if (m_state == PromiseState.Unresolved) {
                    m_resultHandlers.AddLast(handler);
                } else
                    invokeRequired = true;
            }

            // обработчики не должны блокировать сам объект
            if (invokeRequired)
                InvokeHandler(handler);
        }

        void InvokeHandler(ResultHandlerInfo handler) {
            switch (m_state) {
                case PromiseState.Resolved:
                    try {
                        if (handler.resultHandler != null)
                            handler.resultHandler(m_result);
                    } catch (Exception e) {
                        try {
                            if (handler.errorHandler != null)
                                handler.errorHandler(e);
                        } catch { }
                    }
                    break;
                case PromiseState.Rejected:
                    try {
                        if (handler.errorHandler != null)
                            handler.errorHandler(m_error);
                    } catch { }
                    break;
                default:
                    // do nothing
                    return;
            }
        }

        protected virtual void OnStateChanged() {
            switch (m_state) {
                case PromiseState.Resolved:
                    foreach (var resultHandlerInfo in m_resultHandlers)
                        try {
                            if (resultHandlerInfo.resultHandler != null)
                                resultHandlerInfo.resultHandler(m_result);
                        } catch (Exception e) {
                            try {
                                if (resultHandlerInfo.errorHandler != null)
                                    resultHandlerInfo.errorHandler(e);
                            } catch { }
                        }
                    break;
                case PromiseState.Cancelled:
                    foreach (var cancelHandler in m_cancelHandlers)
                        cancelHandler();
                    break;
                case PromiseState.Rejected:
                    foreach (var resultHandlerInfo in m_resultHandlers)
                        try {
                            if (resultHandlerInfo.errorHandler != null)
                                resultHandlerInfo.errorHandler(m_error);
                        } catch { }
                    break;
                default:
                    throw new InvalidOperationException(String.Format("Promise entered an invalid state {0}", m_state));
            }

            m_resultHandlers = null;
            m_cancelHandlers = null;
        }



        public bool IsExclusive {
            get {
                lock (m_lock) {
                    return m_childrenCount <= 1;
                }
            }
        }

        public PromiseState State {
            get {
                lock (m_lock) {
                    return m_state;
                }
            }
        }

        protected bool Cancel(bool dependencies) {
            bool result;

            lock (m_lock) {
                if (m_state == PromiseState.Unresolved) {
                    m_state = PromiseState.Cancelled;
                    result = true;
                } else {
                    result = false;
                }
            }

            if (result)
                OnStateChanged();

            if (dependencies && m_parent != null && m_parent.IsExclusive) {
                m_parent.Cancel();
            }

            return result;
        }

    }
}
