﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Implab {
    public interface IPromise: ICancellable {

        /// <summary>
        /// Тип результата, получаемого через данное обещание.
        /// </summary>
        Type PromiseType { get; }

        /// <summary>
        /// Обещание является выполненым, либо успешно, либо с ошибкой, либо отменено.
        /// </summary>
        bool IsResolved { get; }

        /// <summary>
        /// Обещание было отменено.
        /// </summary>
        bool IsCancelled { get; }

        /// <summary>
        /// Creates a new promise dependend on the current one and resolved on
        /// executing the specified handlers.
        /// </summary>
        /// <param name="success">The handler called on the successful promise completion.</param>
        /// <param name="error">The handler is called if an error while completing the promise occurred.</param>
        /// <param name="cancel">The handler is called in case of promise cancellation.</param>
        /// <returns>The newly created dependant promise.</returns>
        /// <remarks>
        /// <para>
        /// If the success handler is specified the dependend promise will be resolved after the handler is
        /// executed and the dependent promise will be linked to the current one, i.e. the cancellation
        /// of the dependent property will lead to the cancellation of the current promise. If the
        /// success handler isn't specified the dependent promise will not be linked to and
        /// will not be resolved after the successfull resolution of the current one.
        /// </para>
        /// <para>
        /// When the error handler is specified, the exception raised during the current promise completion
        /// will be passed to it as the parameter. If the error handler returns without raising an
        /// exception then the dependant promise will be resolved successfully, otherwise the exception
        /// raised by the handler will be transmitted to the dependent promise. If the handler wants
        /// to passthrough the original exception it needs to wrap the exception with
        /// the <see cref="PromiseTransientException"/>.
        /// </para>
        /// <para>
        /// If the cancelation handler is specified and the current promise is cancelled then the dependent
        /// promise will be resolved after the handler is executed. If the cancelation hendler raises the
        /// exception it will be passed to the dependent promise.
        /// </para>
        /// </remarks>
        IPromise Then(Action success, Action<Exception> error, Action cancel);
        IPromise Then(Action success, Action<Exception> error);
        IPromise Then(Action success);

        IPromise Chain(Func<IPromise> chained, Func<Exception, IPromise> error, Func<IPromise> cancel);
        IPromise Chain(Func<IPromise> chained, Func<Exception, IPromise> error);
        IPromise Chain(Func<IPromise> chained);

        /// <summary>
        /// Adds specified listeners to the current promise.
        /// </summary>
        /// <param name="success">The handler called on the successful promise completion.</param>
        /// <param name="error">The handler is called if an error while completing the promise occurred.</param>
        /// <param name="cancel">The handler is called in case of promise cancellation.</param>
        /// <returns>The current promise.</returns>
        IPromise On(Action success, Action<Exception> error, Action cancel);
        IPromise On(Action success, Action<Exception> error);
        IPromise On(Action success);

        /// <summary>
        /// Adds specified listeners to the current promise.
        /// </summary>
        /// <param name="handler">The handler called on the specified events.</param>
        /// <param name = "events">The combination of flags denoting the events for which the
        /// handler shoud be called.</param>
        /// <returns>The current promise.</returns>
        IPromise On(Action handler, PromiseEventType events);

        /// <summary>
        /// Adds the specified error handler to the current promise
        /// and creates the new dependant promise.
        /// </summary>
        /// <param name="error">
        /// The error handler. If the error handler returns without
        /// an error the dependant promise will be successfully resolved.
        /// </param>
        /// <returns>
        /// The new dependant promise which will be resolved after the error
        /// handler is executed.
        /// </returns>
        /// <remarks>
        /// The successfull result of the current promise will be ignored.
        /// </remarks>
        IPromise Error(Action<Exception> error);

        /// <summary>
        /// Adds the specified cncellation handler to the current promise
        /// and creates the new dependant promise.
        /// </summary>
        /// <returns>
        /// The new dependant promise which will be resolved after the cancellation
        /// handler is executed.
        /// </returns>
        /// <param name="handler">
        /// The cancellation handler.
        /// </param>
        /// <remarks>
        /// If the cancellation handler is executed without an error the dependent
        /// promise will be successfully resolved, otherwise the raised exception
        /// will be passed to the dependant promise. The successful result of the
        /// current promise will be ignored.
        /// </remarks>
        IPromise Cancelled(Action handler);

        /// <summary>
        /// Преобразует результат обещания к заданному типу и возвращает новое обещание.
        /// </summary>
        IPromise<T> Cast<T>();

        /// <summary>
        /// Синхронизирует текущий поток с обещанием.
        /// </summary>
        void Join();
        /// <summary>
        /// Синхронизирует текущий поток с обещанием.
        /// </summary>
        /// <param name="timeout">Время ожидания, по его истечению возникнет исключение.</param>
        /// <exception cref="TimeoutException">Превышено время ожидания.</exception>
        void Join(int timeout);

    }
}
