﻿using System.Threading;
using System;
#if NET_4_5
using System.Threading.Tasks;
#endif

namespace Implab {
    public static class PromiseExtensions {
        public static IPromise<T> DispatchToCurrentContext<T>(this IPromise<T> that) {
            Safe.ArgumentNotNull(that, "that");
            var context = SynchronizationContext.Current;
            if (context == null)
                return that;

            var p = new SyncContextPromise<T>(context, that);

            that.Last(
                p.Resolve,
                p.Reject,
                p.Cancel
            );
            return p;
        }

        public static IPromise<T> DispatchToContext<T>(this IPromise<T> that, SynchronizationContext context) {
            Safe.ArgumentNotNull(that, "that");
            Safe.ArgumentNotNull(context, "context");

            var p = new SyncContextPromise<T>(context, that);

            that.Last(
                p.Resolve,
                p.Reject,
                p.Cancel
            );
            return p;
        }

        /// <summary>
        /// Ensures the dispatched.
        /// </summary>
        /// <returns>The dispatched.</returns>
        /// <param name="that">That.</param>
        /// <param name="head">Head.</param>
        /// <param name="cleanup">Cleanup.</param>
        /// <typeparam name="TPromise">The 1st type parameter.</typeparam>
        /// <typeparam name="T">The 2nd type parameter.</typeparam>
        public static TPromise EnsureDispatched<TPromise,T>(this TPromise that, IPromise<T> head, Action<T> cleanup) where TPromise : IPromise{
            Safe.ArgumentNotNull(that, "that");
            Safe.ArgumentNotNull(head, "head");

            that.Last(null,null,() => head.Last(cleanup));

            return that;
        }

        public static AsyncCallback AsyncCallback<T>(this Promise<T> that, Func<IAsyncResult,T> callback) {
            Safe.ArgumentNotNull(that, "that");
            Safe.ArgumentNotNull(callback, "callback");
            return ar => {
                try {
                    that.Resolve(callback(ar));
                } catch (Exception err) {
                    that.Reject(err);
                }
            };
        }
            
        #if NET_4_5

        public static Task<T> GetTask<T>(this IPromise<T> that) {
            Safe.ArgumentNotNull(that, "that");
            var tcs = new TaskCompletionSource<T>();

            that.Last(tcs.SetResult, tcs.SetException, tcs.SetCanceled);

            return tcs.Task;
        }

        #endif
    }
}

