﻿using System;
using System.Threading;

namespace Implab {
    public class FuncChainTaskBase<TResult> : AbstractPromise<TResult> {
        readonly Func<Exception, IPromise<TResult>> m_error;
        readonly Func<Exception, IPromise<TResult>> m_cancel;

        int m_cancelationLock;

        protected FuncChainTaskBase( Func<Exception, IPromise<TResult>> error, Func<Exception, IPromise<TResult>> cancel, bool autoCancellable) {
            m_error = error;
            m_cancel = cancel;
            if (autoCancellable)
                CancellationRequested(CancelOperation);
        }

        public void Reject(Exception error) {
            if (LockCancelation())
                HandleErrorInternal(error);
        }

        public override void CancelOperation(Exception reason) {
            if (LockCancelation()) {
                if (m_cancel != null) {
                    try {
                        m_cancel(reason).On(SetResult, HandleErrorInternal, SetCancelled);
                    } catch (Exception err) {
                        HandleErrorInternal(err);
                    }
                } else {
                    SetCancelled(reason);
                }
            }

        }

        protected void HandleErrorInternal(Exception error) {
            if (m_error != null) {
                try {
                    var operation = m_error(error);

                    operation.On(SetResult, SetError, SetCancelled);
                    CancellationRequested(operation.Cancel);
                } catch(Exception err) {
                    SetError(err);
                }
            } else {
                SetError(error);
            }
        }

        protected bool LockCancelation() {
            return 0 == Interlocked.CompareExchange(ref m_cancelationLock, 1, 0);
        }
    }
}

