﻿using System;
using System.Threading;

namespace Implab {
    public class FuncTaskBase<TResult> : AbstractPromise<TResult> {
        readonly Func<Exception, TResult> m_cancel;
        readonly Func<Exception, TResult> m_error;

        int m_cancelationLock;

        protected FuncTaskBase( Func<Exception, TResult> error, Func<Exception, TResult> cancel, bool autoCancellable) {
            m_error = error;
            m_cancel = cancel;
            if (autoCancellable)
                CancellationRequested(CancelOperation);
        }

        public void Reject(Exception error) {
            Safe.ArgumentNotNull(error, "error");
            if (LockCancelation())
                HandleErrorInternal(error);
        }

        protected void HandleErrorInternal(Exception error) {
            if (m_error != null) {
                try {
                    SetResult(m_error(error));
                } catch(Exception err) {
                    SetError(err);
                }
            } else {
                SetError(error);
            }
        }

        public override void CancelOperation(Exception reason) {
            if (LockCancelation()) {
                if (m_cancel != null) {
                    try {
                        SetResult(m_cancel(reason));
                    } catch (Exception err) {
                        HandleErrorInternal(err);
                    }
                } else {
                    SetCancelled(reason);
                }
            }
        }

        protected bool LockCancelation() {
            return 0 == Interlocked.CompareExchange(ref m_cancelationLock, 1, 0);
        }
    }
}

