﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Implab.Parsing {
    /// <summary>
    /// Алфавит. Множество символов, которые разбиты на классы, при этом классы имеют непрерывную нумерацию,
    /// что позволяет использовать их в качестве индексов массивов.
    /// </summary>
    /// <remarks>
    /// <para>Алфавит является сюрьективным отображением множества символов в множество индексов, это позволяет сократить размер таблицы переходов автомата
    /// для входных символов, которые для него не различимы.</para>
    /// <para>Далее символами алфавита будем называть классы исходных символов.</para>
    /// </remarks>
    /// <typeparam name="TSymbol">Тип символов.</typeparam>
    public interface IAlphabet<TSymbol> {
        /// <summary>
        /// Количество символов в алфавите.
        /// </summary>
        int Count { get; }
        /// <summary>
        /// Добавляет новый символ в алфавит, если символ уже был добавлен, то
        /// возвращается ранее сопоставленный с символом класс.
        /// </summary>
        /// <param name="symbol">Символ для добавления.</param>
        /// <returns>Индекс класса, который попоставлен с символом.</returns>
        int DefineSymbol(TSymbol symbol);
        /// <summary>
        /// Доабвляем класс символов. Множеству указанных исходных символов 
        /// будет сопоставлен символ в алфавите.
        /// </summary>
        /// <param name="symbols">Множестов исходных символов</param>
        /// <returns>Идентификатор символа алфавита.</returns>
        int DefineClass(IEnumerable<TSymbol> symbols);
        /// <summary>
        /// Создает карту обратного сопоставления символа алфавита и сопоставленным
        /// ему исходным символам.
        /// </summary>
        /// <returns></returns>
        List<TSymbol>[] CreateReverseMap();
        /// <summary>
        /// Создает новый алфавит на основе текущего, горппируя его сиволы в более
        /// крупные непересекающиеся классы символов.
        /// </summary>
        /// <param name="newAlphabet">Новый, пустой алфавит, в котором быдут определены классы.</param>
        /// <param name="classes">Множество классов символов текущего алфавита.</param>
        /// <returns>Карта для перехода символов текущего
        /// алфавита к символам нового.</returns>
        int[] Reclassify(IAlphabet<TSymbol> newAlphabet, IEnumerable<ICollection<int>> classes);

        /// <summary>
        /// Преобразует входной символ в индекс символа из алфавита.
        /// </summary>
        /// <param name="symobl">Исходный символ</param>
        /// <returns>Индекс в алфавите</returns>
        int Translate(TSymbol symobl);
    }
}
