using System;
using System.Diagnostics;

namespace Implab {
    /// <summary>
    /// This class is responsible for the promise resolution, dispatching and chaining
    /// </summary>
    public class Deferred : IResolvable {

        readonly Promise m_promise;
        readonly IDispatcher m_dispatcher;

        internal Deferred(IDispatcher dispatcher) : this(new Promise(), dispatcher) {
        }

        internal Deferred(Promise promise, IDispatcher dispatcher) {
            Debug.Assert(promise != null);
            m_promise = promise;
            m_dispatcher = dispatcher;
        }

        public IPromise Promise {
            get { return m_promise; }
        }

        public void Reject(Exception error) {
            if (error is PromiseTransientException)
                error = ((PromiseTransientException)error).InnerException;

            m_promise.RejectPromise(error);
        }

        public void Resolve() {
            m_promise.ResolvePromise();
        }

        public void Resolve(IPromise thenable) {
            if (thenable == null)
                Reject(new Exception("The promise or task are expected"));
            if (thenable == m_promise)
                Reject(new Exception("The promise cannot be resolved with oneself"));

            else if (m_dispatcher != null)
                // dispatch (see ecma-262/6.0: 25.4.1.3.2 Promise Resolve Functions)
                m_dispatcher.Enqueue(Chain, thenable);
            else
                Chain(thenable);
        }

        void Chain(IPromise thenable) {
            try {
                thenable.Then(this);
            } catch (Exception err) {
                Reject(err);
            }
        }
    }
}