using System;
using Implab.Components;
using Unity;
using Unity.Injection;
using Unity.Registration;

namespace Implab.ServiceHost.Unity
{
    public class FactoryRegistrationBuilder : RegistrationBuilder {
        
        internal InjectionMember Factory { get; private set; }

        internal FactoryRegistrationBuilder(Type registrationType) : base(registrationType) {
        }

        /// <summary>
        /// Задает делегат, который будет использоваться в качестве фабрики
        /// для создания экземпляров, параметры делагата будет заполнены
        /// соответствующими зависимостями.
        /// </summary>
        /// <param name="factory">Фабрика для создания экземпляров.</param>
        public void SetFactoryDelegate(Delegate factory) {
            Safe.ArgumentNotNull(factory, nameof(factory));

            Factory = new DelegateInjectionFactory(factory);
        }

        /// <summary>
        /// Указывает зависимость типа <typeparamref name="T"/> с именем
        /// <paramref name="dependencyName"/>, которая будет передана в качетве
        /// параметра делегату <paramref name="factory"/>
        /// </summary>
        /// <param name="dependencyName">Имя зависимости</param>
        /// <param name="factory">Фабрика для создания экземпляра</param>
        public void SetFactoryDependency<T>(string dependencyName, Func<T, object> factory) {
            Safe.ArgumentNotNull(factory, nameof(factory));

            Factory = new InjectionFactory((c,t,name) => {
                var backend = c.Resolve<T>(dependencyName);
                return factory(backend);
            });
        }

        /// <summary>
        /// Указывает зависимость, реализующую интерфейс <see cref="IFactory{TObj}"/>,
        /// которая будет использоваться в качестве фабрики для создания объектов
        /// </summary>
        /// <param name="dependencyName"></param>
        public void SetFactoryDependency<TFac, TObj>(string dependencyName) where TFac : IFactory<TObj> {
            
            Factory = new InjectionFactory(c => c.Resolve<TFac>(dependencyName).Create());
        }
    }
}