﻿using System;
using System.Threading;

namespace Implab {
    public class ActionChainTaskBase : AbstractPromise {
        readonly Func<Exception, IPromise> m_error;
        readonly Func<Exception, IPromise> m_cancel;

        int m_cancelationLock;

        protected ActionChainTaskBase(Func<Exception, IPromise> error, Func<Exception, IPromise> cancel, bool autoCancellable) {
            m_error = error;
            m_cancel = cancel;
            if (autoCancellable)
                CancellationRequested(CancelOperation);
        }

        public void Reject(Exception error) {
            if (LockCancelation())
                HandleErrorInternal(error);
        }

        public override void CancelOperation(Exception reason) {
            if (LockCancelation()) {
                if (!(reason is OperationCanceledException))
                    reason = reason != null ? new OperationCanceledException(null, reason) : new OperationCanceledException();
                
                if (m_cancel != null) {
                    try {
                        m_cancel(reason).On(SetResult, HandleErrorInternal, HandleCancelInternal);
                    } catch (Exception err) {
                        HandleErrorInternal(err);
                    }
                } else {
                    HandleErrorInternal(reason);
                }
            }
        }

        void HandleCancelInternal(Exception reason) {
            if (!(reason is OperationCanceledException))
                reason = reason != null ? new OperationCanceledException(null, reason) : new OperationCanceledException();
            HandleErrorInternal(reason);
        }

        void HandleErrorInternal(Exception error) {
            if (m_error != null) {
                try {
                    var p = m_error(error);
                    p.On(SetResult, SetError, SetCancelled);
                    CancellationRequested(p.Cancel);
                } catch (Exception err) {
                    error = err;
                }
            } else {
                SetErrorInternal(error);
            }
        }

        void SetErrorInternal(Exception error) {
            while (error is PromiseTransientException)
                error = error.InnerException;

            if (error is OperationCanceledException)
                SetCancelled(error);
            else
                SetError(error);
        }

        protected bool LockCancelation() {
            return 0 == Interlocked.CompareExchange(ref m_cancelationLock, 1, 0);
        }
    }
}

