﻿using Implab.Diagnostics;
using System;
using System.Diagnostics.CodeAnalysis;
using System.Threading;

namespace Implab.Components {
    /// <summary>
    /// Base class the objects which support disposing.
    /// </summary>
    public class Disposable : IDisposable {

        public event EventHandler Disposed;

        public bool IsDisposed {
            get; private set;
        }

        /// <summary>
        /// Asserts the object is not disposed.
        /// </summary>
        /// <exception cref="ObjectDisposedException">The object is disposed</exception>
        /// <remarks>
        protected void AssertNotDisposed() {
            if (IsDisposed)
                throw new ObjectDisposedException(ToString());
        }
        /// <summary>
        /// Вызывает событие <see cref="Disposed"/>
        /// </summary>
        /// <param name="disposing">Признак того, что нужно освободить ресурсы, иначе данный метод
        /// вызван сборщиком мусора и нужно освобождать ТОЛЬКО неуправляемые ресурсы ТОЛЬКО этого
        /// объекта.</param>
        /// <remarks>
        /// Данный метод вызывается гарантированно один раз даже при одновременном вызове <see cref="Dispose()"/>
        /// из нескольких потоков.
        /// </remarks>
        protected virtual void Dispose(bool disposing) {
            if (disposing)
                Disposed.DispatchEvent(this, EventArgs.Empty);
        }

        [SuppressMessage("Microsoft.Design", "CA1063:ImplementIDisposableCorrectly", Justification = "Dipose(bool) and GC.SuppessFinalize are called")]
        public void Dispose() {
            if(!IsDisposed) {
                IsDisposed = true;
                Dispose(true);
                GC.SuppressFinalize(this);
            }
        }

        /// <summary>
        /// Записывает сообщение об утечке объекта.
        /// </summary>
        protected virtual void ReportObjectLeaks() {
            TraceLog.TraceWarning("The object is marked as disposable but isn't disposed properly: {0}", this);
        }

        ~Disposable() {
            Dispose(false);
            ReportObjectLeaks();
        }
    }
}