﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Diagnostics;
using System.Collections;
using System.Runtime.CompilerServices;
using System.Threading.Tasks;
using System.Threading;

#if NET_4_5
using System.Threading.Tasks;
#endif

namespace Implab
{
    public static class Safe
    {
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public static void ArgumentAssert(bool condition, string paramName) {
            if (!condition)
                throw new ArgumentException("The parameter is invalid", paramName);
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public static void ArgumentMatch(string value, string paramName, Regex rx) {
            if (rx == null)
                throw new ArgumentNullException("rx");
            if (!rx.IsMatch(value))
                throw new ArgumentException(String.Format("The prameter value must match {0}", rx), paramName);
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public static void ArgumentNotEmpty(string value, string paramName) {
            if (String.IsNullOrEmpty(value))
                throw new ArgumentException("The parameter can't be empty", paramName);
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public static void ArgumentNotEmpty<T>(T[] value, string paramName) {
            if (value == null || value.Length == 0)
                throw new ArgumentException("The array must be not emty", paramName);
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public static void ArgumentNotNull(object value, string paramName) {
            if (value == null)
                throw new ArgumentNullException(paramName);
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        internal static void ArgumentGreaterEqThan(int value, int min, string paramName) {
            if (value < min)
                throw new ArgumentOutOfRangeException(paramName);
        }

        public static object CreateDefaultValue(Type type) {
            if (type.IsValueType)
                return Activator.CreateInstance(type);

            return null;
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public static void ArgumentInRange(bool condition, string paramName) {
            if (!condition)
                throw new ArgumentOutOfRangeException(paramName);
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public static void ArgumentOfType(object value, Type type, string paramName) {
            if (!type.IsInstanceOfType(value))
                throw new ArgumentException(String.Format("The parameter must be of type {0}", type), paramName);
        }

        public static void Dispose(params IDisposable[] objects) {
            foreach (var d in objects)
                if (d != null)
                    d.Dispose();
        }

        public static void Dispose(params object[] objects) {
            foreach (var obj in objects) {
                var d = obj as IDisposable;
                if (d != null)
                    d.Dispose();
            }
        }

        public static void DisposeCollection(IEnumerable<IDisposable> objects) {
            foreach (var d in objects)
                Dispose(d);
        }

        public static void DisposeCollection(IEnumerable objects) {
            foreach (var d in objects)
                Dispose(d);
        }

        public static void Dispose(object obj) {
            if (obj is IDisposable)
                Dispose((IDisposable)obj);
            
        }

        [DebuggerStepThrough]
        public static void DispatchEvent<T>(this EventHandler<T> handler, object sender, T args) {
            if (handler != null)
                handler(sender, args);
        }

        [DebuggerStepThrough]
        public static void DispatchEvent(this EventHandler handler, object sender, EventArgs args) {
            if (handler != null)
                handler(sender, args);
        }

        [DebuggerStepThrough]
        public static IPromise<T> Run<T>(Func<T> action) {
            ArgumentNotNull(action, "action");

            try {
                return Promise.Resolve(action());
            } catch (Exception err) {
                return Promise.Reject<T>(err);
            }
        }

        [DebuggerStepThrough]
        public static IPromise Run(Action action) {
            ArgumentNotNull(action, "action");

            try {
                action();
                return Promise.Resolve();
            } catch (Exception err) {
                return Promise.Reject(err);
            }
        }

        [DebuggerStepThrough]
        public static IPromise Run(Func<IPromise> action) {
            ArgumentNotNull(action, "action");

            try {
                return action() ?? Promise.Reject(new Exception("The action returned null"));
            } catch (Exception err) {
                return Promise.Reject(err);
            }
        }

        public static void NoWait(IPromise promise) {
        }

        public static void NoWait(Task promise) {
        }

        public static void NoWait<T>(Task<T> promise) {
        }

        public static void Noop() {
        }

        public static void Noop(CancellationToken ct) {
            ct.ThrowIfCancellationRequested();
        }

        public static Task CreateTask() {
            return new Task(Noop);
        }

        public static Task CreateTask(CancellationToken ct) {
            return new Task(Noop, ct);
        }

        [DebuggerStepThrough]
        public static IPromise<T> Run<T>(Func<IPromise<T>> action) {
            ArgumentNotNull(action, "action");

            try {
                return action() ?? Promise.Reject<T>(new Exception("The action returned null"));
            } catch (Exception err) {
                return Promise.Reject<T>(err);
            }
        }

    }
}
