using System;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using Implab.Diagnostics;

namespace Implab.ServiceHost.Unity {
    using System.Linq;
    using System.Reflection;
    using System.Text;
    using global::Unity;
    using global::Unity.Registration;
    using Implab.Xml;
    using static Trace<ContainerBuilder>;

    public class ContainerBuilder {

        readonly TypeResolver m_resolver;

        readonly UnityContainer m_container;

        readonly ContainerConfigurationSchema m_schema;

        public UnityContainer Container {
            get {
                return m_container;
            }
        }

        public ContainerBuilder() : this(null, null) {
        }

        public ContainerBuilder(UnityContainer container, ContainerConfigurationSchema schema) {
            m_container = container ?? new UnityContainer();
            m_resolver = new TypeResolver();
            m_schema = schema ?? ContainerConfigurationSchema.Default;
        }

        public Type ResolveType(string typeReference) {
            return m_resolver.Resolve(typeReference);
        }


        internal void Visit(TypeAbstractRegistration typeRegistration) {
            var registrationType = typeRegistration.GetRegistrationType(ResolveType);
            var implementationType = typeRegistration.GetImplementationType(ResolveType) ?? registrationType;

            var registrationContext = new TypeRegistrationBuilder(
                m_resolver,
                registrationType,
                implementationType
            );

            typeRegistration.Visit(registrationContext);

            m_container.RegisterType(
                registrationContext.RegistrationType,
                registrationContext.ImplementationType,
                typeRegistration.Name,
                registrationContext.Lifetime,
                registrationContext.Injections
            );
        }

        internal void Visit(InstanceAbstractRegistration instanceRegistration) {
            var registrationType = instanceRegistration.GetRegistrationType(ResolveType);

            var builder = new InstanceRegistrationBuilder(m_resolver, registrationType);

            instanceRegistration.Visit(builder);

            m_container.RegisterInstance(
                builder.ValueBuilder.ValueType,
                instanceRegistration.Name,
                builder.ValueBuilder.Value,
                builder.Lifetime
            );
        }

        internal void Visit(FactoryAbstractRegistratrion factoryRgistration) {
            var registrationType = factoryRgistration.GetRegistrationType(ResolveType);

            var builder = new FactoryRegistrationBuilder(registrationType);

            factoryRgistration.Visit(builder);

            m_container.RegisterType(
                builder.RegistrationType,
                factoryRgistration.Name,
                builder.Lifetime,
                builder.Factory
            );
        }

        internal void Visit(NamespaceElement namespaceElement) {
            m_resolver.AddNamespace(namespaceElement.Name);
        }

        internal void Visit(AssemblyElement assemblyElement) {
            Assembly.Load(assemblyElement.AssemblyName);
        }

        internal void Visit(IncludeElement includeElement) {
            Include(includeElement.Href);
        }

        public void Include(string file) {
            var includeContext = new ContainerBuilder(m_container, m_schema);
            includeContext.LoadConfig(file);
        }

        public void LoadConfig(string file) {
            var config = m_schema.LoadFile(file);
            
            config.Visit(this);
        }
        
    }
}