using System;
using System.Diagnostics;
using System.Threading.Tasks;

namespace Implab {
    /// <summary>
    /// This class is responsible for the promise resolution, dispatching and chaining
    /// </summary>
    public class Deferred : IResolvable {

        readonly Promise m_promise;
        internal Deferred() {
            m_promise = new Promise();
        }

        internal Deferred(Promise promise, IDispatcher dispatcher) {
            Debug.Assert(promise != null);
            m_promise = promise;
        }

        public IPromise Promise {
            get { return m_promise; }
        }

        public void Cancel() {
            Reject(new OperationCanceledException());
        }

        public virtual void Reject(Exception error) {
            if (error is PromiseTransientException)
                error = ((PromiseTransientException)error).InnerException;

            m_promise.RejectPromise(error);
        }

        public virtual void Resolve() {
            m_promise.ResolvePromise();
        }

        public virtual void Resolve(IPromise thenable) {
            if (thenable == null)
                Reject(new Exception("The promise or task are expected"));
            if (thenable == m_promise)
                Reject(new Exception("The promise cannot be resolved with oneself"));

            try {
                thenable.Then(this);
            } catch (Exception err) {
                Reject(err);
            }
        }

        public virtual void Resolve(Task thenable) {
            if (thenable == null)
                Reject(new Exception("The promise or task are expected"));
            try {
                thenable.Then(this);
            } catch(Exception err) {
                Reject(err);
            }
        }

    }
}