﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Implab.Diagnostics {
    /// <summary>
    /// Канал, через который публикуются события журнала.
    /// </summary>
    /// <typeparam name="TEvent">Тип событий в канале</typeparam>
    /// <remarks>
    /// Событиями журнала могут быть любые типы, например строки, в которых будет передаваться
    /// информация, или структуры с набором полей, описывающих  важность, текст и другую информацию.
    /// </remarks>
    public class LogChannel<TEvent> {
        static LogChannel<TEvent> _default = new LogChannel<TEvent>();

        /// <summary>
        /// Канал по-умолчанию для событий типа <typeparam name="TEvent"/>.
        /// </summary>
        public static LogChannel<TEvent> Default {
            get {
                return _default;
            }
        }

        /// <summary>
        /// Событие появление новой записи в журнале, на это событие подписываются слушатели.
        /// </summary>
        public event EventHandler<LogEventArgs<TEvent>> Events;

        /// <summary>
        /// Имя канала, полезно для отображения в журнале
        /// </summary>
        public string Name {
            get;
            private set;
        }

        /// <summary>
        /// Создает журнал, имя типа событий назначается в качетве имени канала.
        /// </summary>
        public LogChannel()
            : this(null) {
        }

        /// <summary>
        /// Содает канал с указанным именем.
        /// </summary>
        /// <param name="name">Имя канала.</param>
        public LogChannel(string name) {
            if (String.IsNullOrEmpty(name))
                name = typeof(TEvent).Name;
            Name = name;
        }

        /// <summary>
        /// Отправляет запись журнала через канал подписчикам.
        /// </summary>
        /// <param name="data">Запись журнала.</param>
        /// <remarks>
        /// Контекст трассировки от которого рассылается сообщение определяется автоматически из текущего потока.
        /// </remarks>
        public void LogEvent(TEvent data) {
            var t = Events;
            if (t != null) {
                var traceContext = TraceContext.Instance;
                t(
                    this,
                    new LogEventArgs<TEvent>(
                        data,
                        Name,
                        traceContext.ThreadId,
                        traceContext.CurrentOperation,
                        traceContext.CurrentOperation.Duration
                    )
                );
            }
        }
    }
}
