﻿using Implab;
using Implab.Formats;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Implab.Formats.JSON {
    /// <summary>
    /// Класс для преобразования экранированной строки JSON
    /// </summary>
    public class StringTranslator : TextScanner<JSONGrammar.TokenType> {
        static readonly char[] _escMap;
        static readonly int[] _hexMap;

        static StringTranslator() {
            var chars = new char[] { 'b', 'f', 't', 'r', 'n', '\\', '/' };
            var vals = new char[] { '\b', '\f', '\t', '\r', '\n', '\\', '/' };

            _escMap = new char[chars.Max() + 1];

            for (int i = 0; i < chars.Length; i++)
                _escMap[chars[i]] = vals[i];

            var hexs = new char[] { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f', 'A', 'B', 'C', 'D', 'E', 'F' };
            var ints = new int[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 10, 11, 12, 13, 14, 15 };

            _hexMap = new int[hexs.Max() + 1];

            for (int i = 0; i < hexs.Length; i++)
                _hexMap[hexs[i]] = ints[i];

        }

        public StringTranslator() {
        }

        public string Translate(string data) {
            Safe.ArgumentNotNull(data, "data");
            return Translate(data.ToCharArray());
        }

        public string Translate(char[] data) {
            Safe.ArgumentNotNull(data, "data");
            return Translate(data, data.Length);
        }

        public string Translate(char[] data, int length) {
            Safe.ArgumentNotNull(data, "data");
            Safe.ArgumentInRange(length, 0, data.Length, "length");

            var translated = new char[length];

            Feed(data,length);

            int pos = 0;

            while (ReadTokenInternal()) {
                switch ((JSONGrammar.TokenType)Tags[0]) {
                    case JSONGrammar.TokenType.UnescapedChar:
                        Array.Copy(m_buffer,m_tokenOffset,translated,pos,m_tokenLen);
                        pos += m_tokenLen;
                        break;
                    case JSONGrammar.TokenType.EscapedChar:
                        translated[pos] = _escMap[m_buffer[m_tokenOffset + 1]];
                        pos++;
                        break;
                    case JSONGrammar.TokenType.EscapedUnicode:
                        translated[pos] = TranslateHexUnicode(m_buffer,m_tokenOffset + 2);
                        pos++;
                        break;
                }
            }

            return new String(translated, 0, pos);
        }

        internal static char TranslateEscapedChar(char symbol) {
            return _escMap[symbol];
        }

        internal static char TranslateHexUnicode(char[] symbols, int offset) {
            Debug.Assert(symbols != null);
            Debug.Assert(symbols.Length - offset >= 4);

            int value = (_hexMap[symbols[offset]] << 12)
                | (_hexMap[symbols[offset + 1]] << 8)
                | (_hexMap[symbols[offset + 2]] << 4)
                | (_hexMap[symbols[offset + 3]]);
            return (char)value;
        }
    }
}
