using Implab.Diagnostics;
using System;
using System.Threading;
using System.Linq;

namespace Implab.Parallels {
	/// <summary>
	/// Класс для распаралеливания задач.
	/// </summary>
	/// <remarks>
	/// Используя данный класс и лямда выражения можно распараллелить
	/// вычисления, для этого используется концепция обещаний.
	/// </remarks>
	public static class AsyncPool {

		public static IPromise<T> Invoke<T>(Func<T> func) {
			var p = new Promise<T>();
            var caller = TraceContext.Instance.CurrentOperation;

			ThreadPool.QueueUserWorkItem(param => {
                TraceContext.Instance.EnterLogicalOperation(caller,false);
				try {
					p.Resolve(func());
				} catch(Exception e) {
					p.Reject(e);
                } finally {
                    TraceContext.Instance.Leave();
                }
			});

			return p;
		}

        public static IPromise<T> InvokeNewThread<T>(Func<T> func) {
            var p = new Promise<T>();

            var caller = TraceContext.Instance.CurrentOperation;

            var worker = new Thread(() => {
                TraceContext.Instance.EnterLogicalOperation(caller,false);
                try {
                    p.Resolve(func());
                } catch (Exception e) {
                    p.Reject(e);
                } finally {
                    TraceContext.Instance.Leave();
                }
            });
            worker.IsBackground = true;
            worker.Start();

            return p;
        }


        public static IPromise InvokeNewThread(Action func) {
            var p = new Promise();

            var caller = TraceContext.Instance.CurrentOperation;

            var worker = new Thread(() => {
                TraceContext.Instance.EnterLogicalOperation(caller,false);
                try {
                    func();
                    p.Resolve();
                } catch (Exception e) {
                    p.Reject(e);
                } finally {
                    TraceContext.Instance.Leave();
                }
            });
            worker.IsBackground = true;
            worker.Start();

            return p;
        }

        public static IPromise[] ThreadRun(params Action[] func) {
            return func.Select(f => InvokeNewThread(f)).ToArray();
        }

        public static IPromise<T>[] ThreadRun<T>(params Func<T>[] func) {
            return func.Select(f => InvokeNewThread(f)).ToArray();
        }
	}
}
