using System;
using System.Diagnostics;

namespace Implab {
    class PromiseActionReaction : PromiseReaction {
        readonly Action m_fulfilled;

        readonly Action<Exception> m_rejected;

        readonly Deferred m_next;

        public PromiseActionReaction(Action fulfilled, Action<Exception> rejected, Deferred next, IDispatcher dispatcher) : base(dispatcher) {
            if (fulfilled != null)
                m_fulfilled = () => {
                    fulfilled();
                    next.Resolve();
                };

            if (rejected != null)
                m_rejected = (x) => {
                    rejected(x);
                    next.Resolve();
                };
            m_next = next;
        }

        public PromiseActionReaction(Func<IPromise> fulfilled, Func<Exception, IPromise> rejected, Deferred next, IDispatcher dispatcher) : base(dispatcher) {
            if (fulfilled != null)
                m_fulfilled = () => { next.Resolve(fulfilled()); };
            if (rejected != null)
                m_rejected = (e) => { next.Resolve(rejected(e)); };
            m_next = next;
        }

        public PromiseActionReaction(Action fulfilled, Func<Exception, IPromise> rejected, Deferred next, IDispatcher dispatcher) : base(dispatcher) {
            if (fulfilled != null)
                m_fulfilled = () => {
                    fulfilled();
                    next.Resolve();
                };

            if (rejected != null)
                m_rejected = (e) => { next.Resolve(rejected(e)); };
            m_next = next;
        }

        public PromiseActionReaction(Func<IPromise> fulfilled, Action<Exception> rejected, Deferred next, IDispatcher dispatcher) : base(dispatcher) {
            if (fulfilled != null)
                m_fulfilled = () => { next.Resolve(fulfilled()); };

            if (rejected != null)
                m_rejected = (x) => {
                    rejected(x);
                    next.Resolve();
                };
            m_next = next;
        }


        protected override bool HasFulfilHandler => m_fulfilled != null;

        protected override bool HasRejectHandler => m_rejected != null;

        protected override void DefaultReject(Exception reason) {
            m_next.Reject(reason);
        }

        protected override void DefaultResolve() {
            m_next.Resolve();
        }

        protected override void RejectImpl(Exception reason) {
            try {
                m_rejected(reason);
            } catch (Exception e){
                m_next.Reject(e);
            }
        }

        protected override void ResolveImpl() {
            try {
                m_fulfilled();
            } catch (Exception e){
                m_next.Reject(e);
            }
        }
    }
}