using System.IO;
using System.Text;
using System.Xml;
using System.Xml.Linq;
using System.Xml.Serialization;

namespace Implab.Xml {
    public static class XmlSerializerExtensions {

        public static void Serialize(this XmlSerializer that, object obj, XmlElement element) {
            Safe.ArgumentNotNull(that, nameof(that));
            using (var writer = element.CreateNavigator().AppendChild())
                that.Serialize(writer, obj);
        }

        public static void Serialize(this XmlSerializer that, object obj, XElement element) {
            Safe.ArgumentNotNull(that, nameof(that));
            using (var writer = element.CreateWriter())
                that.Serialize(writer, obj);
        }

        public static XDocument SerializeAsXDocumnet(this XmlSerializer that, object obj) {
            Safe.ArgumentNotNull(that, nameof(that));
            var doc = new XDocument();
            using (var writer = doc.CreateWriter()) {
                that.Serialize(writer, obj);
            }
            return doc;
        }

        public static XmlDocument SerializeAsXmlDocument(this XmlSerializer that, object obj) {
            Safe.ArgumentNotNull(that, nameof(that));
            var doc = new XmlDocument();
            using (var writer = doc.CreateNavigator().AppendChild()) {
                that.Serialize(writer, obj);
            }
            return doc;
        }
        public static string SerializeAsString(this XmlSerializer that, object obj) {
            Safe.ArgumentNotNull(that, nameof(that));
            using (var writer = new StringWriter()) {
                that.Serialize(writer, obj);
                return writer.ToString();
            }
        }

        public static void SerializeToFile(this XmlSerializer that, object obj, string file, Encoding encoding) {
            Safe.ArgumentNotNull(that, nameof(that));
            using (var writer = new StreamWriter(File.Create(file),encoding))
                that.Serialize(writer, obj);
        }

        public static void SerializeToFile(this XmlSerializer that, object obj, string file) {
            SerializeToFile(that, obj, file, Encoding.UTF8);
        }

        public static object Deserialize(this XmlSerializer that, XmlElement element) {
            Safe.ArgumentNotNull(that, nameof(that));
            Safe.ArgumentNotNull(element, nameof(element));

            using (var reader = element.CreateNavigator().ReadSubtree())
                return that.Deserialize(reader);
        }

        public static object Deserialize(this XmlSerializer that, XElement element) {
            Safe.ArgumentNotNull(that, nameof(that));
            Safe.ArgumentNotNull(element, nameof(element));

            using (var reader = element.CreateReader())
                return that.Deserialize(reader);
        }

        public static object DeserializeFromString(this XmlSerializer that, string text) {
            Safe.ArgumentNotNull(that, nameof(that));

            using(var reader = new StringReader(text))
                return that.Deserialize(reader);
        }

        public static object DeserializeFromFile(this XmlSerializer that, string file) {
            Safe.ArgumentNotNull(that, nameof(that));

            using(var reader = File.OpenRead(file))
                return that.Deserialize(reader);
        }
        

    }
}